/**
 * Theme Colors Composable
 * Consistent fitness theme colors across all layouts (Coach, Admin, Client)
 * 
 * Colors from Welcome.vue fitness palette:
 * - Primary: Neon Green (#39FF14)
 * - Secondary: Energetic Orange (#FF6B35)
 * - Accent: Electric Blue (#00D4FF)
 */

import { computed } from 'vue';
import { usePage } from '@inertiajs/vue3';

export function useThemeColors() {
    const page = usePage();
    
    // Get colors from settings or use defaults
    const colors = computed(() => ({
        // Main fitness colors
        primary: page.props.settings?.colors?.primary || '#39FF14',     // Neon green
        secondary: page.props.settings?.colors?.secondary || '#FF6B35', // Energetic orange
        accent: page.props.settings?.colors?.accent || '#00D4FF',       // Electric blue
        
        // Dark theme colors
        dark: '#0A0A0A',
        darkGray: '#1A1A1A',
        metallic: '#2D2D2D',
        
        // Semantic colors
        success: '#39FF14',
        warning: '#FF6B35',
        info: '#00D4FF',
        danger: '#EF4444',
    }));
    
    // CSS classes for common patterns
    const colorClasses = computed(() => ({
        // Primary button (neon green)
        primaryButton: 'bg-neon-500 hover:bg-neon-600 text-black font-semibold shadow-lg shadow-neon-500/25 transition-all duration-200',
        primaryButtonOutline: 'border-2 border-neon-500 text-neon-500 hover:bg-neon-500 hover:text-black transition-all duration-200',
        
        // Secondary button (orange)
        secondaryButton: 'bg-energy-500 hover:bg-energy-600 text-white font-semibold shadow-lg shadow-energy-500/25 transition-all duration-200',
        secondaryButtonOutline: 'border-2 border-energy-500 text-energy-500 hover:bg-energy-500 hover:text-white transition-all duration-200',
        
        // Accent button (electric blue)
        accentButton: 'bg-electric-500 hover:bg-electric-600 text-white font-semibold shadow-lg shadow-electric-500/25 transition-all duration-200',
        accentButtonOutline: 'border-2 border-electric-500 text-electric-500 hover:bg-electric-500 hover:text-white transition-all duration-200',
        
        // Sidebar styles
        sidebarBg: 'bg-fitness-dark dark:bg-fitness-dark',
        sidebarActive: 'bg-neon-500/10 text-neon-500 border-l-4 border-neon-500',
        sidebarHover: 'hover:bg-fitness-dark-50 hover:text-neon-400',
        sidebarText: 'text-gray-300',
        
        // Card styles
        card: 'bg-white dark:bg-fitness-dark-50 border border-gray-200 dark:border-fitness-dark-100 rounded-xl shadow-lg',
        cardHover: 'hover:border-neon-500/50 hover:shadow-neon-500/10 transition-all duration-200',
        
        // Header/navbar
        headerBg: 'bg-white dark:bg-fitness-dark border-b border-gray-200 dark:border-fitness-dark-50',
        
        // Gradients
        gradientPrimary: 'bg-gradient-to-r from-neon-500 to-electric-500',
        gradientSecondary: 'bg-gradient-to-r from-energy-500 to-neon-500',
        gradientDark: 'bg-gradient-to-b from-fitness-dark to-fitness-dark-50',
        
        // Text colors
        textPrimary: 'text-neon-500',
        textSecondary: 'text-energy-500',
        textAccent: 'text-electric-500',
        
        // Badge/tag styles
        badgeSuccess: 'bg-neon-500/20 text-neon-500',
        badgeWarning: 'bg-energy-500/20 text-energy-500',
        badgeInfo: 'bg-electric-500/20 text-electric-500',
    }));
    
    // Inline styles for dynamic color application
    const inlineStyles = computed(() => ({
        primaryGlow: {
            boxShadow: `0 0 20px ${colors.value.primary}40`,
        },
        secondaryGlow: {
            boxShadow: `0 0 20px ${colors.value.secondary}40`,
        },
        accentGlow: {
            boxShadow: `0 0 20px ${colors.value.accent}40`,
        },
    }));
    
    return {
        colors,
        colorClasses,
        inlineStyles,
    };
}
