<?php

namespace Database\Seeders;

use App\Models\Food;
use App\Models\MealItem;
use App\Models\MealPlan;
use App\Models\Workout;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class OfficialTemplatesSeeder extends Seeder
{
    public function run(): void
    {
        $this->seedWorkoutTemplates();
        $this->seedMealPlanTemplates();
    }

    private function seedWorkoutTemplates(): void
    {
        // Don't duplicate if already seeded
        if (Workout::where('is_official', true)->exists()) {
            $this->command->info('Workout templates already seeded, skipping...');
            return;
        }

        $templates = [
            // ============================================================
            // 1. PUSH PULL LEGS - HIPERTROFIA (6 días)
            // ============================================================
            [
                'name' => 'Push Pull Legs - Hipertrofia',
                'description' => 'Rutina clásica de 6 días dividida en empuje, tirón y piernas. Ideal para ganar masa muscular con volumen moderado-alto y descansos controlados.',
                'type' => 'hipertrofia',
                'duration_weeks' => 8,
                'days' => [
                    [
                        'name' => 'Día 1 - Push (Pecho, Hombros, Tríceps)',
                        'exercises' => [
                            ['name' => 'Press de banca con barra', 'muscle_group' => 'pecho', 'sets' => 4, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Controlar la bajada 2-3 seg. Agarre a la anchura de los hombros.', 'block_name' => 'Compuesto Principal', 'order' => 0],
                            ['name' => 'Press inclinado con mancuernas', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Inclinación de 30-45 grados. Sentir estiramiento en el pecho.', 'block_name' => 'Compuesto Principal', 'order' => 1],
                            ['name' => 'Aperturas con mancuernas', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'No bloquear codos. Arco amplio y controlado.', 'block_name' => 'Aislamiento', 'order' => 2],
                            ['name' => 'Press militar con barra', 'muscle_group' => 'hombros', 'sets' => 4, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'De pie o sentado. Core activado.', 'block_name' => 'Compuesto Principal', 'order' => 3],
                            ['name' => 'Elevaciones laterales', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Codos ligeramente flexionados, subir hasta la altura del hombro.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Extensiones de tríceps en polea', 'muscle_group' => 'tríceps', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Codos pegados al cuerpo. Extensión completa.', 'block_name' => 'Aislamiento', 'order' => 5],
                            ['name' => 'Fondos en paralelas', 'muscle_group' => 'tríceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Inclinación frontal para pecho, recto para tríceps.', 'block_name' => 'Aislamiento', 'order' => 6],
                        ],
                    ],
                    [
                        'name' => 'Día 2 - Pull (Espalda, Bíceps)',
                        'exercises' => [
                            ['name' => 'Dominadas o jalón al pecho', 'muscle_group' => 'espalda', 'sets' => 4, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Agarre prono, anchura mayor que hombros. Llevar barra al pecho.', 'block_name' => 'Compuesto Principal', 'order' => 0],
                            ['name' => 'Remo con barra', 'muscle_group' => 'espalda', 'sets' => 4, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Espalda a 45°, llevar barra al ombligo. Apretar escápulas.', 'block_name' => 'Compuesto Principal', 'order' => 1],
                            ['name' => 'Remo con mancuerna a una mano', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Apoyar rodilla en banco. Tirar hacia la cadera.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Pullover con mancuerna', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Brazos semi-extendidos. Sentir estiramiento en dorsales.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Curl con barra', 'muscle_group' => 'bíceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'No balancear. Contracción completa arriba.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Curl martillo', 'muscle_group' => 'bíceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Agarre neutro, trabaja braquiorradial y bíceps.', 'block_name' => 'Aislamiento', 'order' => 5],
                            ['name' => 'Face pulls', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '15-20', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Polea alta, cuerda. Rotar externamente al final.', 'block_name' => 'Aislamiento', 'order' => 6],
                        ],
                    ],
                    [
                        'name' => 'Día 3 - Legs (Piernas y Glúteos)',
                        'exercises' => [
                            ['name' => 'Sentadilla con barra', 'muscle_group' => 'cuádriceps', 'sets' => 4, 'reps' => '8-10', 'rest' => '120', 'rest_seconds' => 120, 'notes' => 'Profundidad: muslos paralelos o más abajo. Rodillas en línea con pies.', 'block_name' => 'Compuesto Principal', 'order' => 0],
                            ['name' => 'Prensa de piernas', 'muscle_group' => 'cuádriceps', 'sets' => 4, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Pies al ancho de hombros. No bloquear rodillas arriba.', 'block_name' => 'Compuesto Principal', 'order' => 1],
                            ['name' => 'Peso muerto rumano', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Barra cerca del cuerpo, sentir estiramiento en isquios.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Extensiones de cuádriceps', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Extensión completa, apretar arriba 1 segundo.', 'block_name' => 'Aislamiento', 'order' => 3],
                            ['name' => 'Curl de piernas acostado', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Mantener caderas pegadas al banco.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Hip thrust con barra', 'muscle_group' => 'glúteos', 'sets' => 4, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Apretar glúteos arriba 2 seg. Mentón al pecho.', 'block_name' => 'Accesorio', 'order' => 5],
                            ['name' => 'Elevaciones de pantorrillas', 'muscle_group' => 'pantorrillas', 'sets' => 4, 'reps' => '15-20', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Rango completo, pausa abajo y arriba.', 'block_name' => 'Aislamiento', 'order' => 6],
                        ],
                    ],
                    [
                        'name' => 'Día 4 - Push (Variación)',
                        'exercises' => [
                            ['name' => 'Press inclinado con barra', 'muscle_group' => 'pecho', 'sets' => 4, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Banco a 30°. Control total en la excéntrica.', 'block_name' => 'Compuesto Principal', 'order' => 0],
                            ['name' => 'Press plano con mancuernas', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Mayor rango de movimiento que con barra.', 'block_name' => 'Compuesto Principal', 'order' => 1],
                            ['name' => 'Cruces en polea alta', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Cruzar manos al frente, apretar pecho.', 'block_name' => 'Aislamiento', 'order' => 2],
                            ['name' => 'Press Arnold', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Rotación completa durante el movimiento.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Elevaciones frontales', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Alternar brazos o simultáneo con mancuernas.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Press francés con barra Z', 'muscle_group' => 'tríceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Bajar detrás de la cabeza, codos apuntando al techo.', 'block_name' => 'Aislamiento', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 5 - Pull (Variación)',
                        'exercises' => [
                            ['name' => 'Remo en máquina o T-bar', 'muscle_group' => 'espalda', 'sets' => 4, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Pecho contra el soporte, tirar hacia el abdomen.', 'block_name' => 'Compuesto Principal', 'order' => 0],
                            ['name' => 'Jalón al pecho agarre cerrado', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Agarre supino cerrado, énfasis en dorsales inferiores.', 'block_name' => 'Compuesto Principal', 'order' => 1],
                            ['name' => 'Remo en polea baja sentado', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'No inclinarse atrás, retraer escápulas.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Encogimientos con mancuernas', 'muscle_group' => 'trapecios', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Subir hombros hacia las orejas, pausar arriba.', 'block_name' => 'Aislamiento', 'order' => 3],
                            ['name' => 'Curl inclinado con mancuernas', 'muscle_group' => 'bíceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Banco a 45°, brazos colgando. Gran estiramiento.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Curl de concentración', 'muscle_group' => 'bíceps', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Codo contra muslo interno. Sin balanceo.', 'block_name' => 'Aislamiento', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 6 - Legs (Variación)',
                        'exercises' => [
                            ['name' => 'Peso muerto convencional', 'muscle_group' => 'espalda baja', 'sets' => 4, 'reps' => '6-8', 'rest' => '150', 'rest_seconds' => 150, 'notes' => 'Espalda recta, empujar con los talones. Ejercicio rey.', 'block_name' => 'Compuesto Principal', 'order' => 0],
                            ['name' => 'Sentadilla búlgara', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Pie trasero en banco. Control de equilibrio.', 'block_name' => 'Accesorio', 'order' => 1],
                            ['name' => 'Zancadas caminando', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '12 cada pierna', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Pasos largos, rodilla trasera casi toca el suelo.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Buenos días con barra', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Barra en trapecios, bisagra de cadera. Peso moderado.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Abducción de cadera en máquina', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '15-20', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Apretar glúteos en la apertura.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Elevaciones de pantorrillas sentado', 'muscle_group' => 'pantorrillas', 'sets' => 4, 'reps' => '15-20', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Trabaja el sóleo. Pausa de 2 seg arriba.', 'block_name' => 'Aislamiento', 'order' => 5],
                        ],
                    ],
                ],
            ],

            // ============================================================
            // 2. FULL BODY 3 DÍAS - PRINCIPIANTES
            // ============================================================
            [
                'name' => 'Full Body 3 Días - Principiantes',
                'description' => 'Rutina ideal para principiantes. 3 días por semana trabajando todo el cuerpo con ejercicios fundamentales. Progresión lineal recomendada.',
                'type' => 'fuerza',
                'duration_weeks' => 12,
                'days' => [
                    [
                        'name' => 'Día 1 - Full Body A',
                        'exercises' => [
                            ['name' => 'Sentadilla con barra', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '8-10', 'rest' => '120', 'rest_seconds' => 120, 'notes' => 'Profundidad paralela. Aumentar peso cada semana 2.5kg.', 'block_name' => 'Compuesto', 'order' => 0],
                            ['name' => 'Press de banca con barra', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Retracción escapular, pies firmes en el suelo.', 'block_name' => 'Compuesto', 'order' => 1],
                            ['name' => 'Remo con barra', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '8-10', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Torso a 45°, tirar a la cadera.', 'block_name' => 'Compuesto', 'order' => 2],
                            ['name' => 'Press militar con mancuernas', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Sentado, control total.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Curl con mancuernas', 'muscle_group' => 'bíceps', 'sets' => 2, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Alternado, supinación completa.', 'block_name' => 'Accesorio', 'order' => 4],
                            ['name' => 'Plancha abdominal', 'muscle_group' => 'abdominales', 'sets' => 3, 'reps' => '30-45 seg', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Cuerpo en línea recta, no dejar caer caderas.', 'block_name' => 'Core', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 2 - Full Body B',
                        'exercises' => [
                            ['name' => 'Peso muerto rumano', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '8-10', 'rest' => '120', 'rest_seconds' => 120, 'notes' => 'Bisagra de cadera, barra cerca de las piernas.', 'block_name' => 'Compuesto', 'order' => 0],
                            ['name' => 'Press inclinado con mancuernas', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Banco a 30°, rango completo.', 'block_name' => 'Compuesto', 'order' => 1],
                            ['name' => 'Jalón al pecho', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Agarre ancho, llevar barra a la clavícula.', 'block_name' => 'Compuesto', 'order' => 2],
                            ['name' => 'Elevaciones laterales', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Codos ligeramente flexionados, hasta la altura del hombro.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Extensiones de tríceps en polea', 'muscle_group' => 'tríceps', 'sets' => 2, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Codos pegados al cuerpo.', 'block_name' => 'Accesorio', 'order' => 4],
                            ['name' => 'Crunch abdominal', 'muscle_group' => 'abdominales', 'sets' => 3, 'reps' => '15-20', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Elevar escápulas del suelo, no tirar del cuello.', 'block_name' => 'Core', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 3 - Full Body C',
                        'exercises' => [
                            ['name' => 'Sentadilla goblet', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Mancuerna pegada al pecho. Ideal para técnica.', 'block_name' => 'Compuesto', 'order' => 0],
                            ['name' => 'Flexiones de brazos', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '10-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Cuerpo recto, bajar hasta que el pecho casi toque el suelo.', 'block_name' => 'Compuesto', 'order' => 1],
                            ['name' => 'Remo con mancuerna', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Una mano apoyada, tirar hacia la cadera.', 'block_name' => 'Compuesto', 'order' => 2],
                            ['name' => 'Hip thrust con barra', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Apretar glúteos 2 seg arriba.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Curl martillo', 'muscle_group' => 'bíceps', 'sets' => 2, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Agarre neutro, alternando brazos.', 'block_name' => 'Accesorio', 'order' => 4],
                            ['name' => 'Elevación de piernas colgando', 'muscle_group' => 'abdominales', 'sets' => 3, 'reps' => '10-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Evitar balanceo. Rodillas al pecho si no puedes recto.', 'block_name' => 'Core', 'order' => 5],
                        ],
                    ],
                ],
            ],

            // ============================================================
            // 3. UPPER LOWER - 4 DÍAS INTERMEDIO
            // ============================================================
            [
                'name' => 'Upper Lower - 4 Días Intermedio',
                'description' => 'Rutina de 4 días dividida en tren superior e inferior. Balance perfecto entre frecuencia y volumen para nivel intermedio.',
                'type' => 'hipertrofia',
                'duration_weeks' => 10,
                'days' => [
                    [
                        'name' => 'Día 1 - Upper (Fuerza)',
                        'exercises' => [
                            ['name' => 'Press de banca con barra', 'muscle_group' => 'pecho', 'sets' => 4, 'reps' => '5-6', 'rest' => '150', 'rest_seconds' => 150, 'notes' => 'Peso pesado, control total. RPE 8-9.', 'block_name' => 'Fuerza', 'order' => 0],
                            ['name' => 'Remo con barra', 'muscle_group' => 'espalda', 'sets' => 4, 'reps' => '5-6', 'rest' => '150', 'rest_seconds' => 150, 'notes' => 'Peso pesado, retracción escapular.', 'block_name' => 'Fuerza', 'order' => 1],
                            ['name' => 'Press militar con barra', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '6-8', 'rest' => '120', 'rest_seconds' => 120, 'notes' => 'Core activado, no arquear la espalda.', 'block_name' => 'Fuerza', 'order' => 2],
                            ['name' => 'Curl con barra', 'muscle_group' => 'bíceps', 'sets' => 3, 'reps' => '8-10', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Sin balanceo, contracción completa.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Extensiones de tríceps sobre la cabeza', 'muscle_group' => 'tríceps', 'sets' => 3, 'reps' => '8-10', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Mancuerna o barra Z.', 'block_name' => 'Accesorio', 'order' => 4],
                            ['name' => 'Face pulls', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '15-20', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Salud de hombros, rotación externa.', 'block_name' => 'Salud', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 2 - Lower (Fuerza)',
                        'exercises' => [
                            ['name' => 'Sentadilla con barra', 'muscle_group' => 'cuádriceps', 'sets' => 4, 'reps' => '5-6', 'rest' => '180', 'rest_seconds' => 180, 'notes' => 'Peso pesado, profundidad completa. RPE 8-9.', 'block_name' => 'Fuerza', 'order' => 0],
                            ['name' => 'Peso muerto rumano', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '8-10', 'rest' => '120', 'rest_seconds' => 120, 'notes' => 'Barra pegada a las piernas. Estiramiento isquios.', 'block_name' => 'Fuerza', 'order' => 1],
                            ['name' => 'Prensa de piernas', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Pies altos para isquios, bajos para cuádriceps.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Hip thrust con barra', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Apretar glúteos en la extensión.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Elevaciones de pantorrillas de pie', 'muscle_group' => 'pantorrillas', 'sets' => 4, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Rango completo, pausa arriba.', 'block_name' => 'Aislamiento', 'order' => 4],
                        ],
                    ],
                    [
                        'name' => 'Día 3 - Upper (Hipertrofia)',
                        'exercises' => [
                            ['name' => 'Press inclinado con mancuernas', 'muscle_group' => 'pecho', 'sets' => 4, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Rango completo, squeeze arriba.', 'block_name' => 'Hipertrofia', 'order' => 0],
                            ['name' => 'Dominadas o jalón al pecho', 'muscle_group' => 'espalda', 'sets' => 4, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Agarre prono, completa extensión abajo.', 'block_name' => 'Hipertrofia', 'order' => 1],
                            ['name' => 'Aperturas con mancuernas', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Arco amplio, sentir estiramiento.', 'block_name' => 'Volumen', 'order' => 2],
                            ['name' => 'Remo en polea baja', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Agarre cerrado, squeeze en escápulas.', 'block_name' => 'Volumen', 'order' => 3],
                            ['name' => 'Elevaciones laterales', 'muscle_group' => 'hombros', 'sets' => 4, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Peso ligero, tensión constante.', 'block_name' => 'Volumen', 'order' => 4],
                            ['name' => 'Curl inclinado + extensiones tríceps (superset)', 'muscle_group' => 'brazos', 'sets' => 3, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Superset: bíceps + tríceps sin descanso entre ellos.', 'block_name' => 'Superset', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 4 - Lower (Hipertrofia)',
                        'exercises' => [
                            ['name' => 'Sentadilla búlgara', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Pie trasero en banco, torso erguido.', 'block_name' => 'Hipertrofia', 'order' => 0],
                            ['name' => 'Curl de piernas acostado', 'muscle_group' => 'isquiotibiales', 'sets' => 4, 'reps' => '12-15', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Caderas pegadas al banco, flexión completa.', 'block_name' => 'Hipertrofia', 'order' => 1],
                            ['name' => 'Extensiones de cuádriceps', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Extensión completa, contraer arriba.', 'block_name' => 'Volumen', 'order' => 2],
                            ['name' => 'Zancadas caminando', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '12 cada pierna', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Pasos largos, activar glúteos.', 'block_name' => 'Volumen', 'order' => 3],
                            ['name' => 'Abducción de cadera', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '15-20', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Apretar glúteos en la apertura.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Elevaciones de pantorrillas sentado', 'muscle_group' => 'pantorrillas', 'sets' => 4, 'reps' => '15-20', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Trabaja el sóleo, rango completo.', 'block_name' => 'Aislamiento', 'order' => 5],
                        ],
                    ],
                ],
            ],

            // ============================================================
            // 4. FUERZA 5x5 - STRONGLIFTS
            // ============================================================
            [
                'name' => 'Fuerza 5x5 - Stronglifts',
                'description' => 'Programa de fuerza basado en el método 5x5. Progresión lineal con los levantamientos compuestos más importantes. Ideal para construir una base de fuerza sólida.',
                'type' => 'fuerza',
                'duration_weeks' => 12,
                'days' => [
                    [
                        'name' => 'Día A - Sentadilla, Press Banca, Remo',
                        'exercises' => [
                            ['name' => 'Sentadilla con barra', 'muscle_group' => 'cuádriceps', 'sets' => 5, 'reps' => '5', 'rest' => '180', 'rest_seconds' => 180, 'notes' => 'Aumentar 2.5kg cada sesión exitosa. Profundidad completa.', 'block_name' => 'Principal', 'order' => 0],
                            ['name' => 'Press de banca con barra', 'muscle_group' => 'pecho', 'sets' => 5, 'reps' => '5', 'rest' => '180', 'rest_seconds' => 180, 'notes' => 'Aumentar 2.5kg cada sesión exitosa. Retracción escapular.', 'block_name' => 'Principal', 'order' => 1],
                            ['name' => 'Remo con barra', 'muscle_group' => 'espalda', 'sets' => 5, 'reps' => '5', 'rest' => '180', 'rest_seconds' => 180, 'notes' => 'Aumentar 2.5kg cada sesión exitosa. Sin impulso.', 'block_name' => 'Principal', 'order' => 2],
                        ],
                    ],
                    [
                        'name' => 'Día B - Sentadilla, Press Militar, Peso Muerto',
                        'exercises' => [
                            ['name' => 'Sentadilla con barra', 'muscle_group' => 'cuádriceps', 'sets' => 5, 'reps' => '5', 'rest' => '180', 'rest_seconds' => 180, 'notes' => 'Se repite cada día. Fundamento del programa.', 'block_name' => 'Principal', 'order' => 0],
                            ['name' => 'Press militar con barra', 'muscle_group' => 'hombros', 'sets' => 5, 'reps' => '5', 'rest' => '180', 'rest_seconds' => 180, 'notes' => 'De pie, core activado. Aumentar 2.5kg por sesión.', 'block_name' => 'Principal', 'order' => 1],
                            ['name' => 'Peso muerto convencional', 'muscle_group' => 'espalda baja', 'sets' => 1, 'reps' => '5', 'rest' => '300', 'rest_seconds' => 300, 'notes' => '1 serie pesada de 5. Aumentar 5kg cada sesión. Forma perfecta.', 'block_name' => 'Principal', 'order' => 2],
                        ],
                    ],
                ],
            ],

            // ============================================================
            // 5. GLÚTEOS Y PIERNAS - FEMENINO
            // ============================================================
            [
                'name' => 'Glúteos y Piernas - Enfoque Femenino',
                'description' => 'Programa de 4 días enfocado en desarrollo de glúteos y piernas con trabajo complementario de tren superior. Incluye activación de glúteos y ejercicios de aislamiento.',
                'type' => 'hipertrofia',
                'duration_weeks' => 8,
                'days' => [
                    [
                        'name' => 'Día 1 - Glúteos (Énfasis)',
                        'exercises' => [
                            ['name' => 'Hip thrust con barra', 'muscle_group' => 'glúteos', 'sets' => 4, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Activar glúteos antes de empezar. Pausa de 2 seg arriba.', 'block_name' => 'Activación', 'order' => 0],
                            ['name' => 'Sentadilla sumo', 'muscle_group' => 'glúteos', 'sets' => 4, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Pies abiertos, puntas afuera. Énfasis en aductores y glúteos.', 'block_name' => 'Compuesto', 'order' => 1],
                            ['name' => 'Peso muerto rumano', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Sentir estiramiento en isquios, apretar glúteos arriba.', 'block_name' => 'Compuesto', 'order' => 2],
                            ['name' => 'Abducción de cadera en máquina', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '15-20', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Inclinarse hacia adelante para glúteo medio.', 'block_name' => 'Aislamiento', 'order' => 3],
                            ['name' => 'Patada de glúteo en polea', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '12-15 c/pierna', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Pierna recta, extensión de cadera controlada.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Puente de glúteos con banda', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '20', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Banda en rodillas, abrir al subir. Finisher.', 'block_name' => 'Finisher', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 2 - Tren Superior',
                        'exercises' => [
                            ['name' => 'Press inclinado con mancuernas', 'muscle_group' => 'pecho', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Banco a 30°, rango completo.', 'block_name' => 'Compuesto', 'order' => 0],
                            ['name' => 'Jalón al pecho', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Agarre ancho, llevar a la clavícula.', 'block_name' => 'Compuesto', 'order' => 1],
                            ['name' => 'Remo con mancuerna', 'muscle_group' => 'espalda', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Tirar hacia la cadera, escápulas juntas.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Elevaciones laterales', 'muscle_group' => 'hombros', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Peso ligero, control total.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Curl con mancuernas', 'muscle_group' => 'bíceps', 'sets' => 2, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Alternado, supinación completa.', 'block_name' => 'Brazos', 'order' => 4],
                            ['name' => 'Extensiones de tríceps en polea', 'muscle_group' => 'tríceps', 'sets' => 2, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Cuerda, extensión completa.', 'block_name' => 'Brazos', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 3 - Piernas (Cuádriceps)',
                        'exercises' => [
                            ['name' => 'Sentadilla con barra', 'muscle_group' => 'cuádriceps', 'sets' => 4, 'reps' => '8-10', 'rest' => '120', 'rest_seconds' => 120, 'notes' => 'Profundidad completa, rodillas sobre los dedos.', 'block_name' => 'Compuesto', 'order' => 0],
                            ['name' => 'Prensa de piernas', 'muscle_group' => 'cuádriceps', 'sets' => 4, 'reps' => '10-12', 'rest' => '90', 'rest_seconds' => 90, 'notes' => 'Pies bajos para más énfasis en cuádriceps.', 'block_name' => 'Compuesto', 'order' => 1],
                            ['name' => 'Sentadilla búlgara', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '10-12 c/pierna', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Pie trasero en banco, torso recto.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Extensiones de cuádriceps', 'muscle_group' => 'cuádriceps', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Extensión completa, apretar 1 seg.', 'block_name' => 'Aislamiento', 'order' => 3],
                            ['name' => 'Curl de piernas', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '12-15', 'rest' => '45', 'rest_seconds' => 45, 'notes' => 'Flexión completa, control en la excéntrica.', 'block_name' => 'Aislamiento', 'order' => 4],
                            ['name' => 'Elevaciones de pantorrillas', 'muscle_group' => 'pantorrillas', 'sets' => 4, 'reps' => '15-20', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Rango completo, pausa arriba.', 'block_name' => 'Aislamiento', 'order' => 5],
                        ],
                    ],
                    [
                        'name' => 'Día 4 - Glúteos (Volumen)',
                        'exercises' => [
                            ['name' => 'Hip thrust con barra', 'muscle_group' => 'glúteos', 'sets' => 4, 'reps' => '12-15', 'rest' => '75', 'rest_seconds' => 75, 'notes' => 'Peso moderado, más repeticiones.', 'block_name' => 'Principal', 'order' => 0],
                            ['name' => 'Step ups con mancuernas', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '12 c/pierna', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Cajón alto a la rodilla, empujar con el talón.', 'block_name' => 'Accesorio', 'order' => 1],
                            ['name' => 'Zancadas inversas', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '12 c/pierna', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Paso atrás, activar glúteos para subir.', 'block_name' => 'Accesorio', 'order' => 2],
                            ['name' => 'Good mornings con barra', 'muscle_group' => 'isquiotibiales', 'sets' => 3, 'reps' => '10-12', 'rest' => '60', 'rest_seconds' => 60, 'notes' => 'Bisagra de cadera, espalda recta.', 'block_name' => 'Accesorio', 'order' => 3],
                            ['name' => 'Caminata lateral con banda', 'muscle_group' => 'glúteos', 'sets' => 3, 'reps' => '15 c/lado', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Semi-sentadilla, pasos controlados.', 'block_name' => 'Finisher', 'order' => 4],
                            ['name' => 'Frog pumps', 'muscle_group' => 'glúteos', 'sets' => 2, 'reps' => '25-30', 'rest' => '30', 'rest_seconds' => 30, 'notes' => 'Plantas de pies juntas, rodillas abiertas. Pumping.', 'block_name' => 'Finisher', 'order' => 5],
                        ],
                    ],
                ],
            ],
        ];

        foreach ($templates as $template) {
            Workout::create(array_merge($template, [
                'user_id' => null,
                'client_id' => null,
                'is_active' => true,
                'is_official' => true,
            ]));
        }

        $this->command->info('✅ 5 workout templates created successfully.');
    }

    private function seedMealPlanTemplates(): void
    {
        if (MealPlan::where('is_official', true)->exists()) {
            $this->command->info('Meal plan templates already seeded, skipping...');
            return;
        }

        // Food IDs map (from DB, lookup by name for safety)
        $foods = Food::whereNull('user_id')->where('is_active', true)->pluck('id', 'name');

        $getFoodId = function (string $name) use ($foods) {
            return $foods->get($name) ?? $foods->first(fn($id, $n) => str_contains(strtolower($n), strtolower($name)));
        };

        // ============================================================
        // 1. DEFINICIÓN / PÉRDIDA DE GRASA - 1800 kcal
        // ============================================================
        $this->createMealPlan(
            'Plan de Definición - 1800 kcal',
            'Plan alimenticio diseñado para pérdida de grasa con alto contenido proteico para preservar masa muscular. Déficit calórico moderado.',
            'lose',
            1800, 160, 150, 50,
            [
                ['meal_type' => 'breakfast', 'items' => [
                    ['name' => 'Huevo entero', 'qty' => 3, 'id' => $getFoodId('Huevo entero')],
                    ['name' => 'Avena', 'qty' => 1, 'id' => $getFoodId('Avena')],
                    ['name' => 'Banana', 'qty' => 0.5, 'id' => $getFoodId('Banana')],
                ]],
                ['meal_type' => 'mid_morning_snack', 'items' => [
                    ['name' => 'Yogur Griego natural', 'qty' => 1, 'id' => $getFoodId('Yogur Griego natural')],
                    ['name' => 'Almendras', 'qty' => 1, 'id' => $getFoodId('Almendras')],
                ]],
                ['meal_type' => 'lunch', 'items' => [
                    ['name' => 'Pechuga de Pollo', 'qty' => 1, 'id' => $getFoodId('Pechuga de Pollo')],
                    ['name' => 'Arroz integral', 'qty' => 1, 'id' => $getFoodId('Arroz integral')],
                    ['name' => 'Brócoli cocido', 'qty' => 1.5, 'id' => $getFoodId('Brócoli cocido')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
                ['meal_type' => 'mid_afternoon_snack', 'items' => [
                    ['name' => 'Proteína de Suero (Whey)', 'qty' => 1, 'id' => $getFoodId('Proteína de Suero (Whey)')],
                    ['name' => 'Manzana', 'qty' => 1, 'id' => $getFoodId('Manzana')],
                ]],
                ['meal_type' => 'dinner', 'items' => [
                    ['name' => 'Salmón', 'qty' => 1, 'id' => $getFoodId('Salmón')],
                    ['name' => 'Espinacas cocidas', 'qty' => 1.5, 'id' => $getFoodId('Espinacas cocidas')],
                    ['name' => 'Batata (Camote)', 'qty' => 0.5, 'id' => $getFoodId('Batata (Camote)')],
                ]],
            ]
        );

        // ============================================================
        // 2. VOLUMEN / GANANCIA MUSCULAR - 3000 kcal
        // ============================================================
        $this->createMealPlan(
            'Plan de Volumen - 3000 kcal',
            'Plan hipercalórico diseñado para ganancia muscular. Superávit calórico controlado con proteínas altas y carbohidratos abundantes.',
            'gain',
            3000, 200, 350, 85,
            [
                ['meal_type' => 'breakfast', 'items' => [
                    ['name' => 'Avena', 'qty' => 2, 'id' => $getFoodId('Avena')],
                    ['name' => 'Huevo entero', 'qty' => 4, 'id' => $getFoodId('Huevo entero')],
                    ['name' => 'Banana', 'qty' => 1, 'id' => $getFoodId('Banana')],
                    ['name' => 'Leche entera', 'qty' => 1, 'id' => $getFoodId('Leche entera')],
                ]],
                ['meal_type' => 'mid_morning_snack', 'items' => [
                    ['name' => 'Pan integral', 'qty' => 2, 'id' => $getFoodId('Pan integral')],
                    ['name' => 'Mantequilla de Maní/Cacahuate', 'qty' => 1, 'id' => $getFoodId('Mantequilla de Maní/Cacahuate')],
                    ['name' => 'Manzana', 'qty' => 1, 'id' => $getFoodId('Manzana')],
                ]],
                ['meal_type' => 'lunch', 'items' => [
                    ['name' => 'Pechuga de Pollo', 'qty' => 1.5, 'id' => $getFoodId('Pechuga de Pollo')],
                    ['name' => 'Arroz blanco cocido', 'qty' => 2, 'id' => $getFoodId('Arroz blanco cocido')],
                    ['name' => 'Aguacate', 'qty' => 0.5, 'id' => $getFoodId('Aguacate')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
                ['meal_type' => 'mid_afternoon_snack', 'items' => [
                    ['name' => 'Proteína de Suero (Whey)', 'qty' => 1, 'id' => $getFoodId('Proteína de Suero (Whey)')],
                    ['name' => 'Almendras', 'qty' => 1, 'id' => $getFoodId('Almendras')],
                    ['name' => 'Banana', 'qty' => 1, 'id' => $getFoodId('Banana')],
                ]],
                ['meal_type' => 'dinner', 'items' => [
                    ['name' => 'Carne de Res magra', 'qty' => 1, 'id' => $getFoodId('Carne de Res magra')],
                    ['name' => 'Batata (Camote)', 'qty' => 1, 'id' => $getFoodId('Batata (Camote)')],
                    ['name' => 'Brócoli cocido', 'qty' => 1, 'id' => $getFoodId('Brócoli cocido')],
                ]],
                ['meal_type' => 'snack', 'items' => [
                    ['name' => 'Yogur Griego natural', 'qty' => 1, 'id' => $getFoodId('Yogur Griego natural')],
                    ['name' => 'Avena', 'qty' => 0.5, 'id' => $getFoodId('Avena')],
                ]],
            ]
        );

        // ============================================================
        // 3. MANTENIMIENTO EQUILIBRADO - 2200 kcal
        // ============================================================
        $this->createMealPlan(
            'Plan Mantenimiento - 2200 kcal',
            'Plan equilibrado para mantener peso con buena distribución de macronutrientes. Ideal para fase de mantenimiento post-déficit o superávit.',
            'maintain',
            2200, 150, 230, 70,
            [
                ['meal_type' => 'breakfast', 'items' => [
                    ['name' => 'Avena', 'qty' => 1.5, 'id' => $getFoodId('Avena')],
                    ['name' => 'Huevo entero', 'qty' => 3, 'id' => $getFoodId('Huevo entero')],
                    ['name' => 'Banana', 'qty' => 1, 'id' => $getFoodId('Banana')],
                ]],
                ['meal_type' => 'lunch', 'items' => [
                    ['name' => 'Tilapia', 'qty' => 1, 'id' => $getFoodId('Tilapia')],
                    ['name' => 'Arroz blanco cocido', 'qty' => 1.5, 'id' => $getFoodId('Arroz blanco cocido')],
                    ['name' => 'Espinacas frescas', 'qty' => 1, 'id' => $getFoodId('Espinacas frescas')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
                ['meal_type' => 'mid_afternoon_snack', 'items' => [
                    ['name' => 'Atún en lata (en agua)', 'qty' => 1, 'id' => $getFoodId('Atún en lata (en agua)')],
                    ['name' => 'Pan integral', 'qty' => 2, 'id' => $getFoodId('Pan integral')],
                    ['name' => 'Aguacate', 'qty' => 0.3, 'id' => $getFoodId('Aguacate')],
                ]],
                ['meal_type' => 'dinner', 'items' => [
                    ['name' => 'Pechuga de Pollo', 'qty' => 1, 'id' => $getFoodId('Pechuga de Pollo')],
                    ['name' => 'Papa (Cocida)', 'qty' => 1, 'id' => $getFoodId('Papa (Cocida)')],
                    ['name' => 'Brócoli cocido', 'qty' => 1, 'id' => $getFoodId('Brócoli cocido')],
                ]],
            ]
        );

        // ============================================================
        // 4. ALTO EN PROTEÍNAS - KETO FRIENDLY - 1600 kcal
        // ============================================================
        $this->createMealPlan(
            'Alto en Proteínas Low-Carb - 1600 kcal',
            'Plan bajo en carbohidratos y alto en proteínas y grasas saludables. Ideal para personas que responden bien a dietas bajas en carbohidratos.',
            'lose',
            1600, 150, 50, 90,
            [
                ['meal_type' => 'breakfast', 'items' => [
                    ['name' => 'Huevo entero', 'qty' => 4, 'id' => $getFoodId('Huevo entero')],
                    ['name' => 'Aguacate', 'qty' => 0.5, 'id' => $getFoodId('Aguacate')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
                ['meal_type' => 'lunch', 'items' => [
                    ['name' => 'Salmón', 'qty' => 1, 'id' => $getFoodId('Salmón')],
                    ['name' => 'Espinacas cocidas', 'qty' => 2, 'id' => $getFoodId('Espinacas cocidas')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
                ['meal_type' => 'mid_afternoon_snack', 'items' => [
                    ['name' => 'Almendras', 'qty' => 1.5, 'id' => $getFoodId('Almendras')],
                ]],
                ['meal_type' => 'dinner', 'items' => [
                    ['name' => 'Pechuga de Pollo', 'qty' => 1, 'id' => $getFoodId('Pechuga de Pollo')],
                    ['name' => 'Brócoli cocido', 'qty' => 2, 'id' => $getFoodId('Brócoli cocido')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
            ]
        );

        // ============================================================
        // 5. PLAN VEGANO - 2000 kcal
        // ============================================================
        $this->createMealPlan(
            'Plan Vegano Balanceado - 2000 kcal',
            'Plan 100% vegetal con buena distribución de macros. Combinaciones estratégicas para obtener proteínas completas de fuentes vegetales.',
            'maintain',
            2000, 100, 260, 65,
            [
                ['meal_type' => 'breakfast', 'items' => [
                    ['name' => 'Avena', 'qty' => 2, 'id' => $getFoodId('Avena')],
                    ['name' => 'Banana', 'qty' => 1, 'id' => $getFoodId('Banana')],
                    ['name' => 'Mantequilla de Maní/Cacahuate', 'qty' => 1, 'id' => $getFoodId('Mantequilla de Maní/Cacahuate')],
                ]],
                ['meal_type' => 'lunch', 'items' => [
                    ['name' => 'Arroz integral', 'qty' => 2, 'id' => $getFoodId('Arroz integral')],
                    ['name' => 'Espinacas cocidas', 'qty' => 1.5, 'id' => $getFoodId('Espinacas cocidas')],
                    ['name' => 'Aguacate', 'qty' => 0.5, 'id' => $getFoodId('Aguacate')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
                ['meal_type' => 'mid_afternoon_snack', 'items' => [
                    ['name' => 'Almendras', 'qty' => 1.5, 'id' => $getFoodId('Almendras')],
                    ['name' => 'Manzana', 'qty' => 1, 'id' => $getFoodId('Manzana')],
                ]],
                ['meal_type' => 'dinner', 'items' => [
                    ['name' => 'Batata (Camote)', 'qty' => 1, 'id' => $getFoodId('Batata (Camote)')],
                    ['name' => 'Brócoli cocido', 'qty' => 1.5, 'id' => $getFoodId('Brócoli cocido')],
                    ['name' => 'Aceite de Oliva', 'qty' => 1, 'id' => $getFoodId('Aceite de Oliva')],
                ]],
            ]
        );

        $this->command->info('✅ 5 meal plan templates created successfully.');
    }

    private function createMealPlan(
        string $name,
        string $description,
        string $goalType,
        int $calories,
        int $proteins,
        int $carbs,
        int $fats,
        array $meals
    ): void {
        DB::beginTransaction();

        try {
            $mealPlan = MealPlan::create([
                'user_id' => null,
                'client_id' => null,
                'name' => $name,
                'description' => $description,
                'goal_type' => $goalType,
                'target_calories' => $calories,
                'target_proteins' => $proteins,
                'target_carbs' => $carbs,
                'target_fats' => $fats,
                'is_active' => true,
                'is_official' => true,
            ]);

            $order = 0;
            foreach ($meals as $meal) {
                foreach ($meal['items'] as $item) {
                    if (!$item['id']) {
                        continue; // Skip if food not found
                    }

                    $food = Food::find($item['id']);
                    if (!$food) continue;

                    $servingWeight = $food->serving_weight ?: 100;
                    $grams = $item['qty'] * $servingWeight;
                    $multiplier = $grams / 100;

                    MealItem::create([
                        'meal_plan_id' => $mealPlan->id,
                        'food_id' => $food->id,
                        'meal_type' => $meal['meal_type'],
                        'quantity' => $grams,
                        'proteins' => round(($food->proteins / ($servingWeight / 100)) * ($grams / 100), 2),
                        'carbs' => round(($food->carbs / ($servingWeight / 100)) * ($grams / 100), 2),
                        'fats' => round(($food->fats / ($servingWeight / 100)) * ($grams / 100), 2),
                        'calories' => round(($food->calories / ($servingWeight / 100)) * ($grams / 100), 2),
                        'order' => $order++,
                    ]);
                }
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error("Error creating meal plan '{$name}': " . $e->getMessage());
        }
    }
}
