<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('coach_payment_methods', function (Blueprint $table) {
            $table->id();
            $table->foreignId('coach_id')->constrained('users')->cascadeOnDelete();
            
            // Type: paypal, bank_dominican, bank_international, western_union
            $table->string('type', 50);
            
            // Display name for the method
            $table->string('name')->nullable();
            
            // Common fields
            $table->boolean('is_active')->default(true);
            $table->boolean('is_verified')->default(false);
            $table->string('verification_token')->nullable();
            $table->timestamp('verified_at')->nullable();
            
            // PayPal fields
            $table->string('paypal_email')->nullable();
            $table->string('paypal_client_id')->nullable();
            $table->string('paypal_secret')->nullable()->comment('Encrypted');
            $table->boolean('paypal_sandbox')->default(false);
            
            // Dominican Bank fields (RD)
            $table->string('bank_name')->nullable();
            $table->string('bank_account_number')->nullable();
            $table->string('bank_account_type')->nullable(); // ahorro, corriente
            $table->string('bank_holder_name')->nullable();
            $table->string('bank_holder_document')->nullable(); // Cédula
            $table->string('bank_holder_document_type')->default('cedula'); // cedula, passport, rnc
            
            // International Bank fields
            $table->string('bank_country')->nullable();
            $table->string('bank_swift')->nullable();
            $table->string('bank_iban')->nullable();
            $table->string('bank_currency', 10)->nullable();
            $table->string('bank_routing_number')->nullable();
            $table->text('bank_additional_info')->nullable();
            
            // Western Union fields
            $table->string('wu_full_name')->nullable();
            $table->string('wu_country')->nullable();
            $table->string('wu_city')->nullable();
            $table->string('wu_phone')->nullable();
            $table->string('wu_id_type')->nullable();
            $table->string('wu_id_number')->nullable();
            
            // Priority for display order
            $table->integer('priority')->default(0);
            
            // Metadata for custom fields
            $table->json('metadata')->nullable();
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['coach_id', 'type', 'is_active']);
            $table->index(['coach_id', 'is_active', 'priority']);
        });

        // Create table for payment receipts/proofs
        Schema::create('payment_receipts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('purchase_id')->constrained('coach_plan_purchases')->cascadeOnDelete();
            $table->foreignId('payment_method_id')->nullable()->constrained('coach_payment_methods')->nullOnDelete();
            
            $table->string('type', 50); // bank_transfer, western_union, paypal
            $table->string('status', 50)->default('pending'); // pending, approved, rejected
            
            // File upload
            $table->string('receipt_image')->nullable();
            
            // Bank transfer specific
            $table->string('transfer_reference')->nullable();
            $table->timestamp('transfer_date')->nullable();
            
            // Western Union specific
            $table->string('mtcn_code', 10)->nullable();
            $table->string('sender_name')->nullable();
            $table->decimal('sent_amount', 12, 2)->nullable();
            $table->string('sent_currency', 10)->nullable();
            
            // PayPal specific
            $table->string('paypal_transaction_id')->nullable();
            $table->string('paypal_payer_id')->nullable();
            $table->string('paypal_payer_email')->nullable();
            
            // Review info
            $table->foreignId('reviewed_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamp('reviewed_at')->nullable();
            $table->text('review_notes')->nullable();
            $table->string('rejection_reason')->nullable();
            
            $table->timestamps();
            
            $table->index(['purchase_id', 'status']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('payment_receipts');
        Schema::dropIfExists('coach_payment_methods');
    }
};
