<?php

namespace App\Services;

use App\Models\EmailLog;
use App\Models\Setting;
use App\Models\User;
use App\Notifications\AdminCriticalEmailFailure;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification as NotificationFacade;

/**
 * Servicio centralizado para gestionar el envío de notificaciones por correo
 * con soporte para logs de auditoría, reintentos automáticos y alertas.
 */
class EmailNotificationService
{
    /**
     * Maximum retry attempts
     */
    protected int $maxRetries = 3;

    /**
     * Threshold for triggering alerts (% failure rate)
     */
    protected float $alertThreshold = 20.0;

    /**
     * Send a notification with full logging
     */
    public function send(
        object $notifiable,
        Notification $notification,
        ?string $subject = null,
        array $metadata = []
    ): ?EmailLog {
        $notificationType = class_basename($notification);
        $subject = $subject ?? $this->extractSubject($notification, $notifiable);

        // Create log entry
        $emailLog = EmailLog::createForNotification(
            notificationType: $notificationType,
            notifiable: $notifiable,
            subject: $subject,
            channel: 'mail',
            metadata: $metadata
        );

        try {
            // Send the notification
            $notifiable->notify($notification);
            
            // Mark as sent
            $emailLog->markAsSent();
            
            Log::info('Email notification sent successfully', [
                'type' => $notificationType,
                'recipient' => $notifiable->email ?? 'unknown',
                'email_log_id' => $emailLog->id,
            ]);
            
            return $emailLog;
        } catch (\Throwable $e) {
            // Mark as failed
            $emailLog->markAsFailed($e->getMessage());
            
            Log::error('Email notification failed', [
                'type' => $notificationType,
                'recipient' => $notifiable->email ?? 'unknown',
                'error' => $e->getMessage(),
                'email_log_id' => $emailLog->id,
            ]);
            
            // Check if we should trigger alerts
            $this->checkAndTriggerAlerts();
            
            return $emailLog;
        }
    }

    /**
     * Retry failed email notifications
     */
    public function retryFailed(): array
    {
        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
        ];

        $failedLogs = EmailLog::forRetry($this->maxRetries)
            ->orderBy('created_at', 'asc')
            ->limit(50) // Process in batches
            ->get();

        foreach ($failedLogs as $emailLog) {
            $results['processed']++;
            
            try {
                // Increment attempt
                $emailLog->incrementAttempts();
                
                // Try to resend based on notification type
                $success = $this->resendNotification($emailLog);
                
                if ($success) {
                    $emailLog->markAsSent();
                    $results['success']++;
                } else {
                    $emailLog->markAsFailed('Resend failed - unknown error');
                    $results['failed']++;
                }
            } catch (\Throwable $e) {
                $emailLog->markAsFailed($e->getMessage());
                $results['failed']++;
                
                Log::error('Email retry failed', [
                    'email_log_id' => $emailLog->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        // Check if we should alert admins
        if ($results['failed'] > 0) {
            $this->checkAndTriggerAlerts();
        }

        return $results;
    }

    /**
     * Attempt to resend a notification based on the log
     */
    protected function resendNotification(EmailLog $emailLog): bool
    {
        // Get the notifiable
        if (!$emailLog->recipient_type || !$emailLog->recipient_id) {
            return false;
        }

        $notifiableClass = $emailLog->recipient_type;
        $notifiable = $notifiableClass::find($emailLog->recipient_id);

        if (!$notifiable) {
            return false;
        }

        // Get notification class from metadata
        $notificationClass = $emailLog->metadata['notification_class'] ?? null;
        
        if (!$notificationClass || !class_exists($notificationClass)) {
            Log::warning('Cannot resend notification - class not found', [
                'email_log_id' => $emailLog->id,
                'notification_class' => $notificationClass,
            ]);
            return false;
        }

        // For simple notifications, we can recreate them
        // This is a basic implementation - complex notifications may need custom handling
        try {
            // Send a generic retry email
            $notifiable->notify(new \App\Notifications\GenericRetryNotification(
                $emailLog->subject,
                $emailLog->metadata['body'] ?? 'Este es un reenvío de una notificación anterior.'
            ));
            
            return true;
        } catch (\Throwable $e) {
            Log::error('Resend notification failed', [
                'email_log_id' => $emailLog->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Check failure rate and trigger alerts if needed
     */
    protected function checkAndTriggerAlerts(): void
    {
        $failureRate = EmailLog::getRecentFailureRate();
        
        if ($failureRate >= $this->alertThreshold) {
            $this->alertAdmins($failureRate);
        }
    }

    /**
     * Alert administrators about email failures
     */
    protected function alertAdmins(float $failureRate): void
    {
        // Check if we already alerted recently (within 1 hour)
        $recentAlert = cache()->get('email_failure_alert_sent');
        
        if ($recentAlert) {
            return; // Don't spam admins
        }

        // Get admins
        $admins = User::where('role', 'admin')->get();
        
        if ($admins->isEmpty()) {
            Log::critical('High email failure rate but no admins to notify', [
                'failure_rate' => $failureRate,
            ]);
            return;
        }

        // Get recent failures for context
        $recentFailures = EmailLog::failed()
            ->recent()
            ->limit(10)
            ->get(['notification_type', 'recipient_email', 'error_message', 'created_at']);

        // Send alert to each admin
        foreach ($admins as $admin) {
            try {
                $admin->notify(new AdminCriticalEmailFailure($failureRate, $recentFailures));
            } catch (\Throwable $e) {
                // Log but don't throw - we don't want alert failures to cause more issues
                Log::error('Failed to send admin alert about email failures', [
                    'admin_id' => $admin->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        // Mark that we've alerted (cache for 1 hour)
        cache()->put('email_failure_alert_sent', now(), 3600);
        
        Log::warning('Admin alerted about high email failure rate', [
            'failure_rate' => $failureRate,
            'admins_notified' => $admins->count(),
        ]);
    }

    /**
     * Extract subject from notification
     */
    protected function extractSubject(Notification $notification, object $notifiable): string
    {
        try {
            $mail = $notification->toMail($notifiable);
            return $mail->subject ?? class_basename($notification);
        } catch (\Throwable $e) {
            return class_basename($notification);
        }
    }

    /**
     * Get email statistics
     */
    public function getStatistics(int $days = 7): array
    {
        return EmailLog::getStatistics($days);
    }

    /**
     * Clean old logs
     */
    public function cleanOldLogs(int $daysToKeep = 30): int
    {
        return EmailLog::where('created_at', '<', now()->subDays($daysToKeep))->delete();
    }
}
