<?php

namespace App\Services;

use App\Models\Order;
use Illuminate\Support\Facades\Config;
use NumberFormatter;

class CurrencyService
{
    /**
     * Supported currencies with their configurations.
     */
    protected array $currencies = [
        'USD' => ['symbol' => '$', 'locale' => 'en_US', 'position' => 'before'],
        'EUR' => ['symbol' => '€', 'locale' => 'es_ES', 'position' => 'after'],
        'GBP' => ['symbol' => '£', 'locale' => 'en_GB', 'position' => 'before'],
        'MXN' => ['symbol' => '$', 'locale' => 'es_MX', 'position' => 'before'],
        'COP' => ['symbol' => '$', 'locale' => 'es_CO', 'position' => 'before'],
        'ARS' => ['symbol' => '$', 'locale' => 'es_AR', 'position' => 'before'],
        'CLP' => ['symbol' => '$', 'locale' => 'es_CL', 'position' => 'before'],
        'PEN' => ['symbol' => 'S/', 'locale' => 'es_PE', 'position' => 'before'],
        'DOP' => ['symbol' => 'RD$', 'locale' => 'es_DO', 'position' => 'before'],
        'BRL' => ['symbol' => 'R$', 'locale' => 'pt_BR', 'position' => 'before'],
    ];

    /**
     * Format an amount with the specified currency.
     *
     * @param float $amount
     * @param string $currencyCode
     * @param string|null $symbol Custom symbol override
     * @return string
     */
    public function format(float $amount, string $currencyCode = 'USD', ?string $symbol = null): string
    {
        $currencyCode = strtoupper($currencyCode);
        $config = $this->currencies[$currencyCode] ?? $this->currencies['USD'];
        
        $symbol = $symbol ?? $config['symbol'];
        $formatted = number_format($amount, 2, '.', ',');
        
        if ($config['position'] === 'after') {
            return "{$formatted}{$symbol}";
        }
        
        return "{$symbol}{$formatted}";
    }

    /**
     * Format an amount using Intl NumberFormatter for proper locale support.
     *
     * @param float $amount
     * @param string $currencyCode
     * @return string
     */
    public function formatWithLocale(float $amount, string $currencyCode = 'USD'): string
    {
        $currencyCode = strtoupper($currencyCode);
        $config = $this->currencies[$currencyCode] ?? $this->currencies['USD'];
        
        if (class_exists(NumberFormatter::class)) {
            $formatter = new NumberFormatter($config['locale'], NumberFormatter::CURRENCY);
            return $formatter->formatCurrency($amount, $currencyCode);
        }
        
        // Fallback to basic formatting
        return $this->format($amount, $currencyCode);
    }

    /**
     * Format an amount for an order (uses order's currency).
     *
     * @param Order $order
     * @param float $amount
     * @return string
     */
    public function formatForOrder(Order $order, float $amount): string
    {
        return $this->format(
            $amount,
            $order->currency ?? 'USD',
            $order->currency_symbol
        );
    }

    /**
     * Get the symbol for a currency code.
     *
     * @param string $currencyCode
     * @return string
     */
    public function getSymbol(string $currencyCode): string
    {
        $currencyCode = strtoupper($currencyCode);
        return $this->currencies[$currencyCode]['symbol'] ?? '$';
    }

    /**
     * Get all supported currencies.
     *
     * @return array
     */
    public function getSupportedCurrencies(): array
    {
        return array_keys($this->currencies);
    }

    /**
     * Get currency configuration.
     *
     * @param string $currencyCode
     * @return array|null
     */
    public function getCurrencyConfig(string $currencyCode): ?array
    {
        $currencyCode = strtoupper($currencyCode);
        return $this->currencies[$currencyCode] ?? null;
    }

    /**
     * Check if a currency is supported.
     *
     * @param string $currencyCode
     * @return bool
     */
    public function isSupported(string $currencyCode): bool
    {
        return isset($this->currencies[strtoupper($currencyCode)]);
    }
}
