<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\Workout;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WorkoutUpdated extends Notification implements ShouldQueue
{
    use Queueable;

    protected $workout;

    public function __construct(Workout $workout)
    {
        $this->workout = $workout;
    }

    public function via($notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $coachName = $this->workout->client?->user?->name ?? 'Tu Coach';

        return (new MailMessage)
            ->subject("💪 Rutina Actualizada - {$this->workout->name}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu coach {$coachName} ha actualizado tu rutina.")
            ->line('')
            ->line('**Detalles de la Rutina:**')
            ->line("Nombre: {$this->workout->name}")
            ->line("Días: " . implode(', ', $this->workout->days ?? []))
            ->action('Ver Mi Rutina', route('client.workout.index'))
            ->line('¡Revisa los cambios en tu plan de entrenamiento!')
            ->salutation("Saludos, {$siteName}");
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'workout_updated',
            'workout_id' => $this->workout->id,
            'workout_name' => $this->workout->name,
            'message' => "Tu coach ha actualizado tu rutina: {$this->workout->name}",
            'icon' => '💪',
            'url' => route('client.workout.index'),
        ];
    }
}
