<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\Workout;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WorkoutAssigned extends Notification implements ShouldQueue
{
    use Queueable;

    protected $workout;

    public function __construct(Workout $workout)
    {
        $this->workout = $workout;
    }

    public function via($notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $coachName = $this->workout->client?->user?->name ?? 'Tu Coach';

        return (new MailMessage)
            ->subject("🏋️ Nueva Rutina Asignada - {$this->workout->name}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu coach {$coachName} te ha asignado una nueva rutina de entrenamiento.")
            ->line('')
            ->line('**Detalles de la Rutina:**')
            ->line("Nombre: {$this->workout->name}")
            ->line("Descripción: " . ($this->workout->description ?? 'Sin descripción'))
            ->action('Ver Mi Rutina', route('client.workout.index'))
            ->line('¡Revisa tu nueva rutina y comienza a entrenar!')
            ->salutation("Saludos, {$siteName}");
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'workout_assigned',
            'workout_id' => $this->workout->id,
            'workout_name' => $this->workout->name,
            'message' => "Tu coach te ha asignado una nueva rutina: {$this->workout->name}",
            'icon' => '🏋️',
            'url' => route('client.workout.index'),
        ];
    }
}
