<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionPlanChanged extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Subscription $subscription,
        public string $oldPlanName,
        public string $newPlanName
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');

        return (new MailMessage)
            ->subject("Cambio de Plan - {$this->newPlanName}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu plan de suscripción ha sido actualizado.")
            ->line("**Plan anterior:** {$this->oldPlanName}")
            ->line("**Nuevo plan:** {$this->newPlanName}")
            ->action('Ver Mi Suscripción', route('coach.subscription.index'))
            ->line("Si tienes alguna pregunta, no dudes en contactarnos.")
            ->salutation("Saludos,\nEl equipo de {$siteName}");
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'subscription_plan_changed',
            'subscription_id' => $this->subscription->id,
            'old_plan' => $this->oldPlanName,
            'new_plan' => $this->newPlanName,
            'message' => "🔄 Plan cambiado: {$this->oldPlanName} → {$this->newPlanName}",
            'title' => 'Plan de suscripción actualizado',
            'url' => route('coach.subscription.index'),
        ];
    }
}
