<?php

namespace App\Notifications;

use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionExpiringSoon extends Notification implements ShouldQueue
{
    use Queueable;

    protected Subscription $subscription;
    protected int $daysRemaining;

    public function __construct(Subscription $subscription, int $daysRemaining)
    {
        $this->subscription = $subscription;
        $this->daysRemaining = $daysRemaining;
    }

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if mail fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Tu suscripción expira pronto')
            ->greeting('Hola ' . $notifiable->name)
            ->line('Tu suscripción al plan **' . $this->subscription->plan->name . '** expira en **' . $this->daysRemaining . ' días**.')
            ->line('Renueva ahora para no perder acceso a tus clientes y herramientas.')
            ->action('Renovar Suscripción', url('/coach/subscription'))
            ->line('¿Tienes preguntas? Contáctanos en soporte@coachpro.com');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'subscription_expiring',
            'subscription_id' => $this->subscription->id,
            'days_remaining' => $this->daysRemaining,
            'message' => '⚠️ Tu suscripción expira en ' . $this->daysRemaining . ' días.',
            'title' => 'Suscripción por expirar',
            'url' => route('coach.subscription.index'),
        ];
    }
}
