<?php

namespace App\Notifications;

use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionActivated extends Notification implements ShouldQueue
{
    use Queueable;

    protected Subscription $subscription;

    public function __construct(Subscription $subscription)
    {
        $this->subscription = $subscription;
    }

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if mail fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('¡Tu suscripción ha sido activada!')
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('Tu suscripción al plan **' . $this->subscription->plan->name . '** ha sido activada exitosamente.')
            ->line('Tu suscripción es válida hasta el ' . $this->subscription->end_date->format('d/m/Y') . '.')
            ->action('Ir a CoachPro', url('/dashboard'))
            ->line('¡Gracias por confiar en CoachPro!');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'subscription_activated',
            'subscription_id' => $this->subscription->id,
            'plan_name' => $this->subscription->plan->name,
            'end_date' => $this->subscription->end_date->toDateString(),
            'message' => '✅ Tu suscripción al plan ' . $this->subscription->plan->name . ' ha sido activada.',
            'title' => 'Suscripción activada',
            'url' => route('coach.subscription.index'),
        ];
    }
}
