<?php

namespace App\Notifications;

use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PaymentReminder extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Subscription $subscription,
        public int $daysRemaining
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $urgencyMessage = $this->daysRemaining <= 3 
            ? '¡Tu suscripción está por vencer muy pronto!' 
            : 'Tu suscripción está próxima a vencer.';

        return (new MailMessage)
            ->subject('Recordatorio de Renovación - ' . $this->daysRemaining . ' días')
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line($urgencyMessage)
            ->line('')
            ->line('**Detalles de tu Suscripción:**')
            ->line('Plan: ' . $this->subscription->plan->name)
            ->line('Vence: ' . $this->subscription->end_date->format('d/m/Y'))
            ->line('Días restantes: ' . $this->daysRemaining)
            ->line('')
            ->line('Para evitar la interrupción del servicio, asegúrate de renovar tu suscripción antes de la fecha de vencimiento.')
            ->action('Renovar Ahora', route('coach.subscription.index'))
            ->line('Gracias por tu preferencia.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'payment_reminder',
            'subscription_id' => $this->subscription->id,
            'plan_name' => $this->subscription->plan->name,
            'end_date' => $this->subscription->end_date->toISOString(),
            'days_remaining' => $this->daysRemaining,
            'url' => route('coach.subscription.index'),
        ];
    }
}
