<?php

namespace App\Notifications;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent when order status changes.
 */
class OrderStatusChanged extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Order $order,
        public string $oldStatus,
        public string $newStatus
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Actualización de Pedido #' . $this->order->order_number)
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('El estado de tu pedido ha sido actualizado.')
            ->line('')
            ->line('**Número de Pedido:** ' . $this->order->order_number)
            ->line('')
            ->line('**Estado Anterior:** ' . $this->getStatusLabel($this->oldStatus))
            ->line('**Nuevo Estado:** ' . $this->getStatusLabel($this->newStatus))
            ->line('')
            ->line($this->getStatusMessage())
            ->action('Ver Detalles del Pedido', $this->getOrderUrl($notifiable))
            ->salutation('Gracias por tu preferencia.');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'order_status_changed',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'old_status' => $this->oldStatus,
            'new_status' => $this->newStatus,
            'message' => "Tu pedido #{$this->order->order_number} ahora está: " . $this->getStatusLabel($this->newStatus),
            'title' => 'Pedido Actualizado',
            'url' => $this->getOrderUrl($notifiable),
        ];
    }

    protected function getStatusLabel(string $status): string
    {
        $labels = [
            'pending' => 'Pendiente',
            'awaiting_payment' => 'Esperando Pago',
            'processing' => 'En Proceso',
            'shipped' => 'Enviado',
            'delivered' => 'Entregado',
            'cancelled' => 'Cancelado',
            'refunded' => 'Reembolsado',
            'paid' => 'Pagado',
        ];

        return $labels[$status] ?? $status;
    }

    protected function getStatusMessage(): string
    {
        return match($this->newStatus) {
            'processing' => 'Tu pedido está siendo preparado para el envío.',
            'shipped' => 'Tu pedido ya está en camino. Pronto recibirás la información de rastreo.',
            'delivered' => '¡Tu pedido ha sido entregado! Esperamos que disfrutes tu compra.',
            'cancelled' => 'Tu pedido ha sido cancelado. Si tienes preguntas, contáctanos.',
            'refunded' => 'Se ha procesado el reembolso de tu pedido.',
            default => 'Revisa los detalles de tu pedido para más información.',
        };
    }

    protected function getOrderUrl(object $notifiable): string
    {
        $prefix = match($notifiable->role) {
            'admin' => 'admin',
            'coach' => 'coach',
            default => 'client',
        };

        return route("{$prefix}.store.orders.show", $this->order->id);
    }

    public function databaseType(object $notifiable): string
    {
        return 'order_status_changed';
    }
}
