<?php

namespace App\Notifications;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to customer when their order is cancelled.
 */
class OrderCancelled extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Order $order,
        public ?string $reason = null
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Pedido Cancelado - #' . $this->order->order_number)
            ->greeting('Hola ' . $notifiable->name)
            ->line('Lamentamos informarte que tu pedido ha sido cancelado.')
            ->line('')
            ->line('**Detalles del Pedido Cancelado:**')
            ->line('• Número de Pedido: ' . $this->order->order_number)
            ->line('• Fecha del Pedido: ' . $this->order->created_at->format('d/m/Y H:i'))
            ->line('• Total: ' . $this->order->formatCurrency($this->order->total));

        if ($this->reason) {
            $message->line('')
                ->line('**Motivo de la Cancelación:**')
                ->line($this->reason);
        }

        if ($this->order->payment_status === 'paid') {
            $message->line('')
                ->line('**Reembolso:**')
                ->line('Ya que el pago había sido procesado, iniciaremos el proceso de reembolso. El tiempo de procesamiento depende de tu método de pago original.');
        }

        $message->line('')
            ->line('Si tienes alguna pregunta o crees que esto fue un error, por favor contáctanos.')
            ->action('Contactar Soporte', url('/page/contact'))
            ->salutation('Disculpa las molestias.');

        return $message;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'order_cancelled',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'total' => $this->order->total,
            'reason' => $this->reason,
            'message' => "Tu pedido #{$this->order->order_number} ha sido cancelado",
            'title' => 'Pedido Cancelado',
            'url' => $this->getOrderUrl($notifiable),
        ];
    }

    protected function getOrderUrl(object $notifiable): string
    {
        $prefix = match($notifiable->role) {
            'admin' => 'admin',
            'coach' => 'coach',
            default => 'client',
        };

        return route("{$prefix}.store.orders.show", $this->order->id);
    }

    public function databaseType(object $notifiable): string
    {
        return 'order_cancelled';
    }
}
