<?php

namespace App\Notifications;

use App\Models\LandingTestimonial;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class NewTestimonialSubmitted extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public LandingTestimonial $testimonial
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if mail fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $coachName = $this->testimonial->coach?->name ?? $this->testimonial->name;
        $coachEmail = $this->testimonial->coach?->email ?? 'N/A';

        return (new MailMessage)
            ->subject('Nuevo testimonio pendiente de revisión')
            ->greeting('Hola Administrador')
            ->line("Un nuevo testimonio ha sido enviado y está pendiente de revisión.")
            ->line("**Coach:** {$coachName}")
            ->line("**Email:** {$coachEmail}")
            ->line("**Calificación:** {$this->testimonial->rating} estrellas")
            ->line("**Testimonio:**")
            ->line("\"{$this->testimonial->content}\"")
            ->action('Revisar Testimonio', route('admin.landing.index') . '?tab=testimonials')
            ->line('Por favor, revisa y aprueba o rechaza el testimonio.');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'new_testimonial_submitted',
            'testimonial_id' => $this->testimonial->id,
            'coach_name' => $this->testimonial->coach?->name ?? $this->testimonial->name,
            'message' => 'Nuevo testimonio pendiente de revisión de ' . ($this->testimonial->coach?->name ?? $this->testimonial->name),
            'url' => route('admin.landing.index') . '?tab=testimonials',
        ];
    }
}
