<?php

namespace App\Notifications;

use App\Models\Message;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class NewMessageNotification extends Notification
{
    // Removed Queueable trait to process notifications immediately
    // This prevents messages from being delayed by queue processing

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Message $message,
        public User $sender
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        // Only database notification for instant delivery
        // Email notifications can be added later if needed
        return ['database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $preview = strlen($this->message->content) > 100
            ? substr($this->message->content, 0, 100) . '...'
            : $this->message->content;

        return (new MailMessage)
            ->subject('Nuevo mensaje de ' . $this->sender->name)
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('Has recibido un nuevo mensaje de **' . $this->sender->name . '**')
            ->line('')
            ->line('**Mensaje:**')
            ->line('"' . $preview . '"')
            ->action('Ver Conversación', $this->getChatUrl($notifiable))
            ->line('Responde directamente desde la plataforma para mantener toda la comunicación organizada.')
            ->salutation('Saludos, ' . config('app.name'));
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'new_message',
            'message_id' => $this->message->id,
            'sender_id' => $this->sender->id,
            'sender_name' => $this->sender->name,
            'sender_avatar' => $this->sender->avatar,
            'content_preview' => substr($this->message->content, 0, 100),
            'created_at' => $this->message->created_at->toISOString(),
            'url' => $this->getChatUrl($notifiable),
        ];
    }

    /**
     * Get the appropriate chat URL based on user role.
     */
    private function getChatUrl(object $notifiable): string
    {
        if ($notifiable->isAdmin()) {
            return route('admin.chat.index', ['coach_id' => $this->sender->id]);
        } elseif ($notifiable->isCoach()) {
            return route('coach.chat.index', ['user' => $this->sender->id]);
        } elseif ($notifiable->isClient()) {
            return route('client.chat.index');
        }

        return route('dashboard');
    }

    /**
     * Determine the notification's database type.
     */
    public function databaseType(object $notifiable): string
    {
        return 'new_message';
    }
}

