<?php

namespace App\Notifications;

use App\Models\Invoice;
use App\Models\Setting;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InvoiceOverdue extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Invoice $invoice,
        public int $daysPastDue = 0
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');

        return (new MailMessage)
            ->subject("⚠️ Factura Vencida - {$this->invoice->invoice_number}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu factura {$this->invoice->invoice_number} está vencida hace {$this->daysPastDue} días.")
            ->line("Monto pendiente: \${$this->invoice->total}")
            ->action('Pagar Ahora', route('coach.subscription.index'))
            ->line('Por favor realiza el pago para mantener tu suscripción activa.')
            ->salutation("Saludos,\nEl equipo de {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'invoice_overdue',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'total' => $this->invoice->total,
            'days_past_due' => $this->daysPastDue,
            'message' => "⚠️ Factura vencida: {$this->invoice->invoice_number} - \${$this->invoice->total} ({$this->daysPastDue} días)",
            'title' => 'Factura vencida',
            'url' => route('coach.subscription.index'),
        ];
    }
}
