<?php

namespace App\Notifications;

use App\Models\Invoice;
use App\Models\Setting;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InvoiceCancelled extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Invoice $invoice,
        public ?string $reason = null
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'DOP');

        $mail = (new MailMessage)
            ->subject("Factura Cancelada - {$this->invoice->invoice_number}")
            ->greeting("Hola {$notifiable->name},")
            ->line("La factura **{$this->invoice->invoice_number}** ha sido cancelada.")
            ->line("**Monto:** \${$this->invoice->total} {$currency}");

        if ($this->reason) {
            $mail->line("**Motivo:** {$this->reason}");
        }

        $mail->action('Ver Mis Facturas', route('coach.subscription.invoices'))
            ->line("Si tienes preguntas, contacta al soporte.")
            ->salutation("Saludos,\nEl equipo de {$siteName}");

        return $mail;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'invoice_cancelled',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'total' => $this->invoice->total,
            'reason' => $this->reason,
            'message' => "❌ Factura {$this->invoice->invoice_number} cancelada",
            'title' => 'Factura cancelada',
            'url' => route('coach.subscription.invoices'),
        ];
    }
}
