<?php

namespace App\Notifications;

use App\Models\Plan;
use App\Models\Setting;
use App\Models\User;
use App\Notifications\Traits\LogsEmailNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notificación de bienvenida para nuevos coaches.
 * 
 * Incluye:
 * - Información de credenciales de acceso
 * - Detalles del plan activo (Trial, Básico, Premium)
 * - Instrucciones para comenzar
 * - Características según el plan
 */
class CoachWelcome extends Notification implements ShouldQueue
{
    use Queueable, LogsEmailNotification;

    public function __construct(
        public User $coach,
        public ?Plan $plan = null,
        public ?string $tempPassword = null
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if mail fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $appName = Setting::get('app_name', 'CoachPro');
        $planName = $this->plan?->name ?? 'Trial';
        $planSlug = $this->plan?->slug ?? 'trial';
        
        $subject = "🎉 ¡Bienvenido a {$appName}, {$this->coach->name}!";

        // Create email log
        $this->createEmailLog($notifiable, $subject, [
            'coach_id' => $this->coach->id,
            'plan_id' => $this->plan?->id,
            'plan_slug' => $planSlug,
        ]);

        $mail = (new MailMessage)
            ->subject($subject)
            ->greeting("¡Hola {$this->coach->name}!")
            ->line("Te damos la bienvenida a **{$appName}**, la plataforma profesional para entrenadores personales.")
            ->line('')
            ->line('---')
            ->line('## 🔐 Tus Credenciales de Acceso')
            ->line('')
            ->line("**Email:** {$this->coach->email}");

        // Only show temporary password if provided (for admin-created accounts)
        if ($this->tempPassword) {
            $mail->line("**Contraseña temporal:** {$this->tempPassword}")
                 ->line('⚠️ Por seguridad, te recomendamos cambiar tu contraseña después de iniciar sesión.');
        }

        $mail->line('')
             ->line('---')
             ->line("## 📋 Tu Plan: {$planName}");

        // Add plan-specific content
        $mail = $this->addPlanContent($mail, $planSlug);

        $mail->line('')
             ->line('---')
             ->line('## 🚀 Primeros Pasos')
             ->line('')
             ->line('1. **Inicia sesión** en tu panel de coach')
             ->line('2. **Completa tu perfil** profesional')
             ->line('3. **Agrega tu primer cliente**')
             ->line('4. **Crea rutinas y planes de alimentación**');

        $mail->action('Acceder a Mi Panel', url('/dashboard'))
             ->line('')
             ->line('---')
             ->line("¿Tienes dudas? Visita nuestra sección de soporte o contáctanos directamente.")
             ->salutation("¡Éxito en tu carrera como coach!\nEl equipo de {$appName}");

        return $mail;
    }

    /**
     * Add plan-specific content to the email
     */
    protected function addPlanContent(MailMessage $mail, string $planSlug): MailMessage
    {
        switch ($planSlug) {
            case 'trial':
                return $this->addTrialContent($mail);
            case 'basico':
                return $this->addBasicContent($mail);
            case 'premium':
                return $this->addPremiumContent($mail);
            default:
                return $this->addTrialContent($mail);
        }
    }

    /**
     * Content for Trial plan
     */
    protected function addTrialContent(MailMessage $mail): MailMessage
    {
        $durationDays = $this->plan?->duration_days ?? 7;
        $features = $this->plan?->features ?? [];
        $maxClients = $features['max_clients'] ?? 3;

        return $mail
            ->line('')
            ->line("🎁 **Plan de Prueba Gratuito** - {$durationDays} días")
            ->line('')
            ->line('**Incluye:**')
            ->line("✅ Hasta {$maxClients} clientes activos")
            ->line('✅ Creación de rutinas de entrenamiento')
            ->line('✅ Planes de alimentación básicos')
            ->line('✅ Chat con tus clientes')
            ->line('✅ Portal de cliente')
            ->line('')
            ->line('**Limitaciones:**')
            ->line('❌ Sin marca personalizada (branding)')
            ->line('❌ Sin exportación de PDFs')
            ->line('❌ Sin tienda de productos')
            ->line('')
            ->line("⏰ **Tu período de prueba termina en {$durationDays} días.** Actualiza a un plan de pago para continuar sin interrupciones.");
    }

    /**
     * Content for Basic plan
     */
    protected function addBasicContent(MailMessage $mail): MailMessage
    {
        $features = $this->plan?->features ?? [];
        $maxClients = $features['max_clients'] ?? 15;
        $price = $this->plan?->price ?? '9.99';

        return $mail
            ->line('')
            ->line("💼 **Plan Básico** - \${$price}/mes")
            ->line('')
            ->line('**Incluye todo del Trial más:**')
            ->line("✅ Hasta {$maxClients} clientes activos")
            ->line('✅ Exportación de PDFs con tu marca')
            ->line('✅ Biblioteca de ejercicios personalizada')
            ->line('✅ Biblioteca de alimentos personalizada')
            ->line('✅ Historial completo de clientes')
            ->line('✅ Soporte por email prioritario')
            ->line('')
            ->line('**Limitaciones:**')
            ->line('❌ Sin tienda de productos')
            ->line('❌ Sin planes de coach para venta')
            ->line('')
            ->line('💡 **Consejo:** El Plan Básico es perfecto para coaches que están comenzando su carrera profesional.');
    }

    /**
     * Content for Premium plan
     */
    protected function addPremiumContent(MailMessage $mail): MailMessage
    {
        $features = $this->plan?->features ?? [];
        $maxClients = $features['max_clients'] ?? -1;
        $clientsText = $maxClients === -1 ? 'Clientes ilimitados' : "Hasta {$maxClients} clientes";
        $price = $this->plan?->price ?? '29.99';

        return $mail
            ->line('')
            ->line("👑 **Plan Premium** - \${$price}/mes")
            ->line('')
            ->line('**Incluye todo del Básico más:**')
            ->line("✅ {$clientsText}")
            ->line('✅ Tienda de productos integrada')
            ->line('✅ Vende tus propios planes de entrenamiento')
            ->line('✅ Sistema de cupones y descuentos')
            ->line('✅ Programa de referidos con comisiones')
            ->line('✅ Personalización completa de marca')
            ->line('✅ Reportes y analytics avanzados')
            ->line('✅ Soporte prioritario 24/7')
            ->line('✅ Wallet para gestionar ganancias')
            ->line('')
            ->line('🌟 **¡Felicidades!** Tienes acceso completo a todas las funcionalidades de la plataforma.');
    }

    /**
     * Get metadata for the email log
     */
    protected function getEmailLogMetadata(): array
    {
        return [
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'coach_email' => $this->coach->email,
            'plan_id' => $this->plan?->id,
            'plan_slug' => $this->plan?->slug,
            'plan_name' => $this->plan?->name,
            'has_temp_password' => !empty($this->tempPassword),
        ];
    }

    public function toArray(object $notifiable): array
    {
        $appName = Setting::get('app_name', 'CoachPro');
        $planName = $this->plan?->name ?? 'Trial';

        return [
            'type' => 'coach_welcome',
            'coach_id' => $this->coach->id,
            'plan_name' => $planName,
            'message' => "🎉 ¡Bienvenido a {$appName}! Tu plan {$planName} está activo.",
            'title' => 'Bienvenido',
            'url' => route('dashboard'),
        ];
    }
}
