<?php

namespace App\Notifications;

use App\Models\Client;
use App\Models\Setting;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminNewClientCreated extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Client $client,
        public User $coach
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');

        return (new MailMessage)
            ->subject("👤 Nuevo Cliente Registrado - {$this->coach->name}")
            ->greeting('¡Hola Administrador!')
            ->line("El coach {$this->coach->name} ha registrado un nuevo cliente en la plataforma.")
            ->line('')
            ->line('**Detalles del Cliente:**')
            ->line("Nombre: {$this->client->name}")
            ->line("Email: " . ($this->client->email ?? 'No especificado'))
            ->line("Coach: {$this->coach->name}")
            ->line("Fecha: " . now()->format('d/m/Y H:i'))
            ->action('Ver Cliente', route('admin.users.show', $this->coach->id))
            ->line('El coach ahora puede asignar rutinas y planes de alimentación a este cliente.')
            ->salutation("Saludos, {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_new_client_created',
            'client_id' => $this->client->id,
            'client_name' => $this->client->name,
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'message' => "👤 Nuevo cliente: {$this->client->name} registrado por {$this->coach->name}",
            'title' => 'Nuevo cliente registrado',
            'url' => route('admin.users.show', $this->coach->id),
        ];
    }
}
