<?php

namespace App\Notifications;

use App\Models\MealPlan;
use App\Models\Setting;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminMealPlanCreated extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public MealPlan $mealPlan,
        public User $coach
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $clientName = $this->mealPlan->client?->name ?? 'Cliente';

        return (new MailMessage)
            ->subject("🍽️ Nuevo Plan Alimenticio - {$this->coach->name}")
            ->greeting('¡Hola Administrador!')
            ->line("El coach {$this->coach->name} ha creado un nuevo plan de alimentación.")
            ->line('')
            ->line('**Detalles del Plan:**')
            ->line("Nombre: {$this->mealPlan->name}")
            ->line("Cliente: {$clientName}")
            ->line("Coach: {$this->coach->name}")
            ->line("Fecha: " . now()->format('d/m/Y H:i'))
            ->action('Ver Coach', route('admin.users.show', $this->coach->id))
            ->line('El plan de alimentación ya está disponible para el cliente.')
            ->salutation("Saludos, {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_meal_plan_created',
            'meal_plan_id' => $this->mealPlan->id,
            'meal_plan_name' => $this->mealPlan->name,
            'client_name' => $this->mealPlan->client?->name ?? 'Cliente',
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'message' => "🍽️ Nuevo plan: {$this->mealPlan->name} creado por {$this->coach->name}",
            'title' => 'Nuevo plan de alimentación',
            'url' => route('admin.users.show', $this->coach->id),
        ];
    }
}
