<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CoachEmailTemplate extends Model
{
    protected $fillable = [
        'coach_id',
        'type',
        'subject',
        'body',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Template types with their display names and available variables.
     */
    public static array $templateTypes = [
        'purchase_confirmed' => [
            'name' => 'Compra Confirmada',
            'description' => 'Se envía cuando confirmas el pago de una compra',
            'variables' => ['buyer_name', 'plan_name', 'amount', 'ends_at', 'coach_name'],
            'default_subject' => '✅ ¡Pago confirmado! - {{plan_name}}',
            'default_body' => '<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
    <div style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
        <h1 style="color: white; margin: 0; font-size: 24px;">¡Pago Confirmado!</h1>
    </div>
    <div style="background: #ffffff; padding: 30px; border: 1px solid #e5e7eb;">
        <p style="font-size: 16px; color: #374151;">¡Hola <strong>{{buyer_name}}</strong>!</p>
        <p style="color: #6b7280;">Tu pago para el plan <strong>{{plan_name}}</strong> ha sido confirmado.</p>
        <div style="background: #f3f4f6; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <p style="margin: 5px 0; color: #374151;"><strong>Monto:</strong> ${{amount}}</p>
            <p style="margin: 5px 0; color: #374151;"><strong>Válido hasta:</strong> {{ends_at}}</p>
        </div>
        <p style="color: #6b7280;">Gracias por tu confianza.</p>
        <p style="color: #374151; font-weight: bold;">{{coach_name}}</p>
    </div>
</div>',
        ],
        'purchase_pending' => [
            'name' => 'Compra Pendiente',
            'description' => 'Se envía cuando alguien compra tu plan (esperando pago)',
            'variables' => ['buyer_name', 'plan_name', 'amount', 'payment_instructions', 'coach_name'],
            'default_subject' => '🛒 Compra recibida - {{plan_name}}',
            'default_body' => '<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
    <div style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
        <h1 style="color: white; margin: 0; font-size: 24px;">Compra Recibida</h1>
    </div>
    <div style="background: #ffffff; padding: 30px; border: 1px solid #e5e7eb;">
        <p style="font-size: 16px; color: #374151;">¡Hola <strong>{{buyer_name}}</strong>!</p>
        <p style="color: #6b7280;">Hemos recibido tu compra del plan <strong>{{plan_name}}</strong>.</p>
        <div style="background: #fef3c7; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <h3 style="margin: 0 0 10px 0; color: #92400e;">Instrucciones de pago:</h3>
            <p style="color: #92400e; margin: 0;">{{payment_instructions}}</p>
            <p style="margin: 10px 0 0 0; color: #374151;"><strong>Monto a pagar:</strong> ${{amount}}</p>
        </div>
        <p style="color: #6b7280;">Una vez recibido el pago, confirmaremos tu acceso.</p>
        <p style="color: #374151; font-weight: bold;">{{coach_name}}</p>
    </div>
</div>',
        ],
        'payment_reminder' => [
            'name' => 'Recordatorio de Pago',
            'description' => 'Se envía como recordatorio de pago pendiente',
            'variables' => ['buyer_name', 'plan_name', 'amount', 'payment_instructions', 'coach_name'],
            'default_subject' => '⏰ Recordatorio de pago - {{plan_name}}',
            'default_body' => '<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
    <div style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
        <h1 style="color: white; margin: 0; font-size: 24px;">Recordatorio de Pago</h1>
    </div>
    <div style="background: #ffffff; padding: 30px; border: 1px solid #e5e7eb;">
        <p style="font-size: 16px; color: #374151;">¡Hola <strong>{{buyer_name}}</strong>!</p>
        <p style="color: #6b7280;">Te recordamos que tu compra del plan <strong>{{plan_name}}</strong> está pendiente de pago.</p>
        <div style="background: #f3f4f6; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <p style="margin: 5px 0; color: #374151;"><strong>Monto:</strong> ${{amount}}</p>
            <p style="margin: 10px 0 0 0; color: #6b7280;">{{payment_instructions}}</p>
        </div>
        <p style="color: #6b7280;">¿Tienes alguna pregunta? Responde a este email.</p>
        <p style="color: #374151; font-weight: bold;">{{coach_name}}</p>
    </div>
</div>',
        ],
        'plan_expiring' => [
            'name' => 'Plan por Expirar',
            'description' => 'Se envía días antes de que expire el plan',
            'variables' => ['buyer_name', 'plan_name', 'days_remaining', 'ends_at', 'coach_name'],
            'default_subject' => '⚠️ Tu plan expira pronto - {{plan_name}}',
            'default_body' => '<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
    <div style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
        <h1 style="color: white; margin: 0; font-size: 24px;">Tu Plan Expira Pronto</h1>
    </div>
    <div style="background: #ffffff; padding: 30px; border: 1px solid #e5e7eb;">
        <p style="font-size: 16px; color: #374151;">¡Hola <strong>{{buyer_name}}</strong>!</p>
        <p style="color: #6b7280;">Tu plan <strong>{{plan_name}}</strong> expira en <strong>{{days_remaining}} días</strong>.</p>
        <div style="background: #fef3c7; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <p style="margin: 0; color: #92400e;"><strong>Fecha de expiración:</strong> {{ends_at}}</p>
        </div>
        <p style="color: #6b7280;">Si deseas renovar, contáctame para continuar con tu entrenamiento.</p>
        <p style="color: #374151; font-weight: bold;">{{coach_name}}</p>
    </div>
</div>',
        ],
        'plan_expired' => [
            'name' => 'Plan Expirado',
            'description' => 'Se envía cuando el plan ha expirado',
            'variables' => ['buyer_name', 'plan_name', 'coach_name'],
            'default_subject' => '📅 Tu plan ha expirado - {{plan_name}}',
            'default_body' => '<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
    <div style="background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
        <h1 style="color: white; margin: 0; font-size: 24px;">Plan Expirado</h1>
    </div>
    <div style="background: #ffffff; padding: 30px; border: 1px solid #e5e7eb;">
        <p style="font-size: 16px; color: #374151;">¡Hola <strong>{{buyer_name}}</strong>!</p>
        <p style="color: #6b7280;">Tu plan <strong>{{plan_name}}</strong> ha expirado.</p>
        <p style="color: #6b7280;">Fue un placer acompañarte en tu entrenamiento. Si deseas continuar, estaré encantado de ayudarte.</p>
        <p style="color: #374151; font-weight: bold;">{{coach_name}}</p>
    </div>
</div>',
        ],
        'plan_suspended' => [
            'name' => 'Plan Suspendido',
            'description' => 'Se envía cuando suspendes el acceso de un cliente',
            'variables' => ['buyer_name', 'plan_name', 'coach_name'],
            'default_subject' => '🔒 Acceso suspendido - {{plan_name}}',
            'default_body' => '<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
    <div style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
        <h1 style="color: white; margin: 0; font-size: 24px;">Acceso Suspendido</h1>
    </div>
    <div style="background: #ffffff; padding: 30px; border: 1px solid #e5e7eb;">
        <p style="font-size: 16px; color: #374151;">Hola <strong>{{buyer_name}}</strong>,</p>
        <p style="color: #6b7280;">Tu acceso al plan <strong>{{plan_name}}</strong> ha sido suspendido.</p>
        <p style="color: #6b7280;">Si crees que esto es un error, por favor contáctame.</p>
        <p style="color: #374151; font-weight: bold;">{{coach_name}}</p>
    </div>
</div>',
        ],
    ];

    /**
     * Get the coach that owns this template.
     */
    public function coach(): BelongsTo
    {
        return $this->belongsTo(User::class, 'coach_id');
    }

    /**
     * Get template by type for a specific coach.
     * Returns coach's custom template or null if not set.
     */
    public static function getForCoach(int $coachId, string $type): ?self
    {
        return static::where('coach_id', $coachId)
            ->where('type', $type)
            ->where('is_active', true)
            ->first();
    }

    /**
     * Render the template with given data.
     */
    public function render(array $data = []): array
    {
        $subject = $this->subject;
        $body = $this->body;

        foreach ($data as $key => $value) {
            $subject = str_replace("{{{$key}}}", $value ?? '', $subject);
            $body = str_replace("{{{$key}}}", $value ?? '', $body);
        }

        return [
            'subject' => $subject,
            'body' => $body,
        ];
    }

    /**
     * Get default template content for a type.
     */
    public static function getDefaultTemplate(string $type): ?array
    {
        return self::$templateTypes[$type] ?? null;
    }
}
