<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Client extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'client_user_id',
        'name',
        'email',
        'phone',
        'age',
        'weight', // KG
        'height', // CM
        'gender',
        'objective',
        'activity_level',
        'calorie_adjustment',
        'bmr',
        'tdee',
        'bmi',
        'notes',
        'avatar',
        'is_active',
        'portal_active',
        'portal_password',
    ];

    protected $casts = [
        'weight' => 'decimal:2',
        'height' => 'decimal:2',
        'bmr' => 'decimal:2',
        'tdee' => 'decimal:2',
        'bmi' => 'decimal:2',
        'calorie_adjustment' => 'integer',
        'is_active' => 'boolean',
        'portal_active' => 'boolean',
    ];

    /**
     * Activity level factors for TDEE calculation.
     * Sedentario 1.2x, Moderada 1.375x, Activo 1.55x, Muy activo 1.725x
     */
    const ACTIVITY_FACTORS = [
        'sedentary' => 1.2,
        'light' => 1.375,
        'moderate' => 1.55,
        'active' => 1.725,
        'very_active' => 1.9,
    ];

    /**
     * Get the coach (user) for this client.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Alias for user() - Get the coach for this client.
     */
    public function coach(): BelongsTo
    {
        return $this->user();
    }

    /**
     * Get the portal user account for this client.
     */
    public function portalUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'client_user_id');
    }

    /**
     * Get workouts for this client.
     */
    public function workouts(): HasMany
    {
        return $this->hasMany(Workout::class);
    }

    /**
     * Get meal plans for this client.
     */
    public function mealPlans(): HasMany
    {
        return $this->hasMany(MealPlan::class);
    }

    /**
     * Get compliance logs for this client.
     */
    public function complianceLogs(): HasMany
    {
        return $this->hasMany(ComplianceLog::class);
    }

    /**
     * Get progress photos for this client.
     */
    public function progressPhotos(): HasMany
    {
        return $this->hasMany(ProgressPhoto::class);
    }

    /**
     * Calculate BMR using Mifflin-St Jeor equation (TMB in Spanish).
     */
    public function calculateBmr(): ?float
    {
        if (!$this->weight || !$this->height || !$this->age || !$this->gender) {
            return null;
        }

        if ($this->gender === 'male') {
            return (10 * $this->weight) + (6.25 * $this->height) - (5 * $this->age) + 5;
        }

        return (10 * $this->weight) + (6.25 * $this->height) - (5 * $this->age) - 161;
    }

    /**
     * Calculate TDEE (Gasto Energético Total).
     */
    public function calculateTdee(): ?float
    {
        $bmr = $this->calculateBmr();

        if (!$bmr) {
            return null;
        }

        $factor = self::ACTIVITY_FACTORS[$this->activity_level] ?? 1.2;

        return round($bmr * $factor, 2);
    }

    /**
     * Calculate BMI (IMC: Índice de Masa Corporal).
     * Formula: kg / m^2
     */
    public function calculateBmi(): ?float
    {
        if (!$this->weight || !$this->height) {
            return null;
        }

        $heightInMeters = $this->height / 100;
        return round($this->weight / ($heightInMeters * $heightInMeters), 2);
    }

    /**
     * Calculate recommended macros based on TDEE and objective.
     * Returns [protein_g, carbs_g, fats_g, calories]
     */
    public function calculateMacros(): ?array
    {
        $tdee = $this->tdee ?? $this->calculateTdee();
        
        if (!$tdee) {
            return null;
        }

        // Adjust calories based on objective
        $targetCalories = match($this->objective) {
            'lose' => $tdee * 0.8, // 20% deficit
            'gain' => $tdee * 1.15, // 15% surplus
            default => $tdee, // maintain
        };

        // Protein: 2g per kg of body weight (common for athletes)
        $proteinGrams = $this->weight ? round($this->weight * 2, 0) : round($targetCalories * 0.25 / 4, 0);
        $proteinCalories = $proteinGrams * 4;

        // Fat: 25% of total calories
        $fatCalories = $targetCalories * 0.25;
        $fatGrams = round($fatCalories / 9, 0);

        // Carbs: remaining calories
        $carbCalories = $targetCalories - $proteinCalories - $fatCalories;
        $carbGrams = round($carbCalories / 4, 0);

        return [
            'calories' => round($targetCalories, 0),
            'protein' => max(0, $proteinGrams),
            'carbs' => max(0, $carbGrams),
            'fats' => max(0, $fatGrams),
        ];
    }

    /**
     * Get formatted macros for display.
     */
    public function getMacrosAttribute(): ?array
    {
        return $this->calculateMacros();
    }

    /**
     * Update all metrics.
     */
    public function updateMetrics(): void
    {
        $this->bmr = $this->calculateBmr();
        $this->tdee = $this->calculateTdee();
        $this->bmi = $this->calculateBmi();
        $this->save();
    }

    // Unit Helpers
    public function getWeightLbsAttribute(): float
    {
        return round($this->weight * 2.20462, 2);
    }

    public function getHeightFtAttribute(): float
    {
        return round($this->height * 0.0328084, 2);
    }

    /**
     * Scope for active clients.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
}
