<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use App\Models\ReferralCode;
use App\Models\Referral;
use App\Models\Setting;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Notifications\AdminWalletTransactionPending;
use App\Notifications\WalletDepositConfirmed;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Inertia\Response;
use Stripe\Checkout\Session as StripeSession;
use Stripe\Stripe;

class WalletController extends Controller
{
    public function index(Request $request): Response
    {
        $user = $request->user();
        $wallet = $user->wallet ?? Wallet::create(['user_id' => $user->id]);
        
        // Get or create referral code
        $referralCode = ReferralCode::getOrCreateForUser($user);
        
        // Get referral stats
        $referralStats = [
            'total' => Referral::where('referrer_id', $user->id)->count(),
            'pending' => Referral::where('referrer_id', $user->id)->pending()->count(),
            'rewarded' => Referral::where('referrer_id', $user->id)->rewarded()->count(),
            'total_earned' => Referral::where('referrer_id', $user->id)
                ->rewarded()
                ->sum('reward_amount'),
            'clicks' => $referralCode->clicks,
        ];

        // Get recent referrals
        $referrals = Referral::with('referred:id,name,email,created_at')
            ->where('referrer_id', $user->id)
            ->latest()
            ->take(10)
            ->get();

        // Get payment methods available for recharge
        $paymentMethods = $this->getAvailablePaymentMethods();

        return Inertia::render('Coach/Wallet/Index', [
            'wallet' => $wallet,
            'transactions' => $wallet->transactions()->paginate(20),
            'referral_code' => $referralCode->code,
            'referral_url' => route('register', ['ref' => $referralCode->code]),
            'referral_stats' => $referralStats,
            'referrals' => $referrals,
            'payment_methods' => $paymentMethods,
        ]);
    }

    protected function getAvailablePaymentMethods(): array
    {
        $methods = [];
        
        // Check Stripe - use 'stripe_secret' (the key used in settings)
        if (Setting::get('stripe_enabled') && Setting::get('stripe_secret')) {
            $methods[] = [
                'id' => 'stripe',
                'name' => 'Tarjeta de Crédito/Débito',
                'description' => 'Pago seguro con Stripe',
                'icon' => 'credit-card',
            ];
        }
        
        // Check PayPal - use 'paypal_secret' (the key used in settings)
        if (Setting::get('paypal_enabled') && Setting::get('paypal_client_id')) {
            $methods[] = [
                'id' => 'paypal',
                'name' => 'PayPal',
                'description' => 'Paga con tu cuenta PayPal',
                'icon' => 'paypal',
            ];
        }
        
        // Bank Transfer
        $banks = Bank::where('is_active', true)->get();
        if ($banks->count() > 0) {
            $methods[] = [
                'id' => 'bank_transfer',
                'name' => 'Transferencia Bancaria',
                'description' => 'Depósito o transferencia manual',
                'icon' => 'bank',
                'banks' => $banks,
            ];
        }

        return $methods;
    }
    
    /**
     * Get decrypted Stripe secret key from settings.
     */
    protected function getStripeSecret(): ?string
    {
        $encrypted = Setting::get('stripe_secret');
        if (!$encrypted) {
            return null;
        }
        try {
            return decrypt($encrypted);
        } catch (\Exception $e) {
            // If not encrypted, return as is (for backward compatibility)
            return $encrypted;
        }
    }
    
    /**
     * Get decrypted PayPal secret from settings.
     */
    protected function getPayPalSecret(): ?string
    {
        $encrypted = Setting::get('paypal_secret');
        if (!$encrypted) {
            return null;
        }
        try {
            return decrypt($encrypted);
        } catch (\Exception $e) {
            // If not encrypted, return as is
            return $encrypted;
        }
    }

    public function recharge(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:5|max:10000',
            'payment_method' => 'required|in:stripe,paypal,bank_transfer',
            'bank_id' => 'required_if:payment_method,bank_transfer|nullable|exists:banks,id',
            'transfer_reference' => 'required_if:payment_method,bank_transfer|nullable|string|max:255',
        ]);

        $user = $request->user();
        $wallet = $user->wallet ?? Wallet::create(['user_id' => $user->id]);
        $amount = (float) $validated['amount'];

        try {
            switch ($validated['payment_method']) {
                case 'stripe':
                    return $this->processStripeRecharge($wallet, $amount);
                    
                case 'paypal':
                    return $this->processPayPalRecharge($wallet, $amount);
                    
                case 'bank_transfer':
                    return $this->processBankRecharge($wallet, $amount, $validated);
            }
        } catch (\Exception $e) {
            Log::error('Wallet recharge error: ' . $e->getMessage());
            return back()->with('error', 'Error al procesar la recarga: ' . $e->getMessage());
        }
    }

    protected function processStripeRecharge(Wallet $wallet, float $amount)
    {
        $stripeSecret = $this->getStripeSecret();
        if (!$stripeSecret) {
            throw new \Exception('Stripe no está configurado correctamente.');
        }
        
        // Apply Stripe fee if enabled (2.9% + $0.30)
        $feeEnabled = Setting::get('stripe_fee_enabled', false);
        $chargeAmount = $amount;
        $feeAmount = 0;
        
        if ($feeEnabled) {
            // Calculate fee: (amount + 0.30) / (1 - 0.029)
            $chargeAmount = round(($amount + 0.30) / (1 - 0.029), 2);
            $feeAmount = round($chargeAmount - $amount, 2);
        }
        
        Stripe::setApiKey($stripeSecret);

        $session = StripeSession::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                    'currency' => strtolower($wallet->currency ?? 'usd'),
                    'product_data' => [
                        'name' => 'Recarga de Saldo',
                        'description' => $feeEnabled 
                            ? 'Recarga de $' . number_format($amount, 2) . ' (incluye comisión de procesamiento: $' . number_format($feeAmount, 2) . ')'
                            : 'Recarga de $' . number_format($amount, 2) . ' a tu billetera',
                    ],
                    'unit_amount' => (int) ($chargeAmount * 100),
                ],
                'quantity' => 1,
            ]],
            'mode' => 'payment',
            'success_url' => route('coach.wallet.stripe.success') . '?session_id={CHECKOUT_SESSION_ID}',
            'cancel_url' => route('coach.wallet.stripe.cancel'),
            'metadata' => [
                'wallet_id' => $wallet->id,
                'amount' => $amount, // The actual amount to credit
                'charge_amount' => $chargeAmount,
                'fee_amount' => $feeAmount,
                'type' => 'wallet_recharge',
            ],
        ]);

        // Create pending transaction (credit the requested amount, not the charged amount)
        $wallet->createPendingDeposit($amount, 'stripe', $session->id, [
            'stripe_session_id' => $session->id,
            'charge_amount' => $chargeAmount,
            'fee_amount' => $feeAmount,
        ]);

        return Inertia::location($session->url);
    }

    protected function processPayPalRecharge(Wallet $wallet, float $amount)
    {
        // Create PayPal order
        $clientId = Setting::get('paypal_client_id');
        $clientSecret = $this->getPayPalSecret();
        $isSandbox = Setting::get('paypal_sandbox', true);
        
        if (!$clientId || !$clientSecret) {
            throw new \Exception('PayPal no está configurado correctamente. Por favor configure las credenciales en Configuración > Pagos.');
        }
        
        // Apply PayPal fee if enabled (3.49% + $0.49)
        $feeEnabled = Setting::get('paypal_fee_enabled', false);
        $chargeAmount = $amount;
        $feeAmount = 0;
        
        if ($feeEnabled) {
            // Calculate fee: (amount + 0.49) / (1 - 0.0349)
            $chargeAmount = round(($amount + 0.49) / (1 - 0.0349), 2);
            $feeAmount = round($chargeAmount - $amount, 2);
        }
        
        // Log para debug (sin exponer datos sensibles)
        Log::info('PayPal recharge attempt', [
            'wallet_id' => $wallet->id,
            'amount' => $amount,
            'charge_amount' => $chargeAmount,
            'fee_amount' => $feeAmount,
            'sandbox' => $isSandbox,
            'client_id_prefix' => substr($clientId, 0, 10) . '...',
        ]);
        
        $baseUrl = $isSandbox 
            ? 'https://api-m.sandbox.paypal.com' 
            : 'https://api-m.paypal.com';

        // Get access token
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "$baseUrl/v1/oauth2/token");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials');
        curl_setopt($ch, CURLOPT_USERPWD, "$clientId:$clientSecret");
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            Log::error('PayPal CURL error', ['error' => $curlError]);
            throw new \Exception('Error de conexión con PayPal: ' . $curlError);
        }

        $tokenData = json_decode($response, true);
        
        if (!isset($tokenData['access_token'])) {
            Log::error('PayPal token error', [
                'http_code' => $httpCode,
                'response' => $tokenData,
            ]);
            
            $errorMsg = $tokenData['error_description'] ?? ($tokenData['error'] ?? 'Error de autenticación');
            throw new \Exception('Error al conectar con PayPal: ' . $errorMsg . '. Verifique las credenciales en Configuración > Pagos.');
        }

        // Create order with fee-adjusted amount
        $orderData = [
            'intent' => 'CAPTURE',
            'purchase_units' => [[
                'amount' => [
                    'currency_code' => strtoupper($wallet->currency ?? 'USD'),
                    'value' => number_format($chargeAmount, 2, '.', ''),
                ],
                'description' => $feeEnabled 
                    ? 'Recarga de saldo ($' . number_format($amount, 2) . ' + comisión: $' . number_format($feeAmount, 2) . ')'
                    : 'Recarga de saldo',
            ]],
            'application_context' => [
                'return_url' => route('coach.wallet.paypal.success'),
                'cancel_url' => route('coach.wallet.paypal.cancel'),
            ],
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "$baseUrl/v2/checkout/orders");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($orderData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $tokenData['access_token'],
        ]);
        $response = curl_exec($ch);
        curl_close($ch);

        $order = json_decode($response, true);

        if (!isset($order['id'])) {
            throw new \Exception('Error al crear orden de PayPal');
        }

        // Create pending transaction (credit the requested amount, not the charged amount)
        $wallet->createPendingDeposit($amount, 'paypal', $order['id'], [
            'paypal_order_id' => $order['id'],
            'charge_amount' => $chargeAmount,
            'fee_amount' => $feeAmount,
        ]);

        // Find approval URL
        $approvalUrl = collect($order['links'])->firstWhere('rel', 'approve')['href'] ?? null;

        if (!$approvalUrl) {
            throw new \Exception('No se pudo obtener URL de aprobación de PayPal');
        }

        return Inertia::location($approvalUrl);
    }

    protected function processBankRecharge(Wallet $wallet, float $amount, array $data)
    {
        $bank = Bank::findOrFail($data['bank_id']);
        
        // Create pending transaction awaiting admin approval
        $transaction = $wallet->createPendingDeposit($amount, 'bank_transfer', $data['transfer_reference'], [
            'bank_id' => $bank->id,
            'bank_name' => $bank->bank_name,
            'transfer_reference' => $data['transfer_reference'],
        ]);

        // Notify admins about pending bank deposit
        $admins = \App\Models\User::where('role', 'admin')->get();
        \Illuminate\Support\Facades\Notification::send($admins, new AdminWalletTransactionPending($transaction));

        return back()->with('success', 'Tu solicitud de recarga ha sido enviada. Una vez verificado el pago, se acreditará a tu cuenta.');
    }

    public function stripeSuccess(Request $request)
    {
        $sessionId = $request->get('session_id');
        
        if (!$sessionId) {
            return redirect()->route('coach.wallet.index')
                ->with('error', 'Sesión de pago no encontrada.');
        }

        try {
            $stripeSecret = $this->getStripeSecret();
            if (!$stripeSecret) {
                throw new \Exception('Stripe no está configurado.');
            }
            
            Stripe::setApiKey($stripeSecret);
            $session = StripeSession::retrieve($sessionId);

            if ($session->payment_status === 'paid') {
                $wallet = Wallet::findOrFail($session->metadata->wallet_id);
                
                // Find the pending transaction
                $transaction = WalletTransaction::where('payment_reference', $sessionId)
                    ->where('status', 'pending')
                    ->first();

                if ($transaction) {
                    $wallet->confirmTransaction($transaction);

                    // Notify coach of successful deposit
                    $wallet->user->notify(new WalletDepositConfirmed($transaction->fresh()));
                }

                return redirect()->route('coach.wallet.index')
                    ->with('success', '¡Recarga exitosa! Tu saldo ha sido actualizado.');
            }
        } catch (\Exception $e) {
            Log::error('Stripe wallet success error: ' . $e->getMessage());
        }

        return redirect()->route('coach.wallet.index')
            ->with('error', 'Error al procesar el pago.');
    }

    public function stripeCancel()
    {
        return redirect()->route('coach.wallet.index')
            ->with('warning', 'Recarga cancelada.');
    }

    public function paypalSuccess(Request $request)
    {
        $token = $request->get('token');
        
        if (!$token) {
            return redirect()->route('coach.wallet.index')
                ->with('error', 'Token de pago no encontrado.');
        }

        try {
            // Capture the payment
            $clientId = Setting::get('paypal_client_id');
            $clientSecret = $this->getPayPalSecret();
            $isSandbox = Setting::get('paypal_sandbox', true);
            
            if (!$clientId || !$clientSecret) {
                throw new \Exception('PayPal no está configurado.');
            }
            
            $baseUrl = $isSandbox 
                ? 'https://api-m.sandbox.paypal.com' 
                : 'https://api-m.paypal.com';

            // Get access token
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, "$baseUrl/v1/oauth2/token");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials');
            curl_setopt($ch, CURLOPT_USERPWD, "$clientId:$clientSecret");
            $response = curl_exec($ch);
            curl_close($ch);

            $tokenData = json_decode($response, true);

            // Capture the order
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, "$baseUrl/v2/checkout/orders/$token/capture");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, '');
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $tokenData['access_token'],
            ]);
            $response = curl_exec($ch);
            curl_close($ch);

            $capture = json_decode($response, true);

            if (isset($capture['status']) && $capture['status'] === 'COMPLETED') {
                // Find and confirm the transaction
                $transaction = WalletTransaction::where('payment_reference', $token)
                    ->where('status', 'pending')
                    ->first();

                if ($transaction) {
                    $wallet = $transaction->wallet;
                    $wallet->confirmTransaction($transaction);

                    // Notify coach of successful deposit
                    $wallet->user->notify(new WalletDepositConfirmed($transaction->fresh()));
                }

                return redirect()->route('coach.wallet.index')
                    ->with('success', '¡Recarga exitosa! Tu saldo ha sido actualizado.');
            }
        } catch (\Exception $e) {
            Log::error('PayPal wallet success error: ' . $e->getMessage());
        }

        return redirect()->route('coach.wallet.index')
            ->with('error', 'Error al procesar el pago de PayPal.');
    }

    public function paypalCancel()
    {
        return redirect()->route('coach.wallet.index')
            ->with('warning', 'Recarga cancelada.');
    }

    public function withdraw(Request $request): \Illuminate\Http\RedirectResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:10',
            'details' => 'required|string|max:500',
        ]);

        $user = $request->user();
        $wallet = $user->wallet;

        if (!$wallet || $wallet->balance < $request->amount) {
            return back()->with('error', 'Saldo insuficiente.');
        }

        try {
            $transaction = $wallet->withdraw(
                (float) $request->amount,
                'Solicitud de retiro: ' . $request->details,
                ['payment_details' => $request->details]
            );

            // Notify admins about pending withdrawal
            $admins = \App\Models\User::where('role', 'admin')->get();
            \Illuminate\Support\Facades\Notification::send($admins, new AdminWalletTransactionPending($transaction));

            return back()->with('success', 'Solicitud de retiro enviada. Te notificaremos cuando sea procesada.');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }
}
