<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Post;
use App\Models\Setting;
use App\Models\Tag;
use Illuminate\Http\Request;
use Inertia\Inertia;

class BlogController extends Controller
{
    /**
     * Get common data shared across all blog pages.
     */
    protected function getCommonData(): array
    {
        return [
            'systemLogo' => Setting::get('logo'),
            'siteName' => Setting::get('site_name', 'CoachPro'),
        ];
    }

    public function index(Request $request)
    {
        $posts = Post::with(['author', 'category', 'tags'])
            ->published()
            ->when($request->search, fn($q, $search) => $q->search($search))
            ->latest('published_at')
            ->paginate(12)
            ->withQueryString();

        return Inertia::render('Blog/Index', array_merge($this->getCommonData(), [
            'posts' => $posts,
            'categories' => Category::withCount('publishedPosts')->get(),
            'popularTags' => Tag::withCount('publishedPosts')
                ->having('published_posts_count', '>', 0)
                ->orderByDesc('published_posts_count')
                ->limit(10)
                ->get(),
            'filters' => $request->only(['search']),
        ]));
    }

    public function show(string $slug)
    {
        $post = Post::with(['author', 'category', 'tags'])
            ->where('slug', $slug)
            ->published()
            ->firstOrFail();

        // Increment views
        $post->incrementViews();

        // Related posts
        $relatedPosts = Post::with(['category'])
            ->published()
            ->where('id', '!=', $post->id)
            ->where(function($q) use ($post) {
                $q->where('category_id', $post->category_id)
                  ->orWhereHas('tags', function($tq) use ($post) {
                      $tq->whereIn('tags.id', $post->tags->pluck('id'));
                  });
            })
            ->latest('published_at')
            ->limit(3)
            ->get();

        return Inertia::render('Blog/Show', array_merge($this->getCommonData(), [
            'post' => $post,
            'relatedPosts' => $relatedPosts,
        ]));
    }

    public function category(string $slug)
    {
        $category = Category::where('slug', $slug)->firstOrFail();

        $posts = Post::with(['author', 'category', 'tags'])
            ->published()
            ->where('category_id', $category->id)
            ->latest('published_at')
            ->paginate(12);

        return Inertia::render('Blog/Category', array_merge($this->getCommonData(), [
            'category' => $category,
            'posts' => $posts,
            'categories' => Category::withCount('publishedPosts')->get(),
        ]));
    }

    public function tag(string $slug)
    {
        $tag = Tag::where('slug', $slug)->firstOrFail();

        $posts = Post::with(['author', 'category', 'tags'])
            ->published()
            ->whereHas('tags', fn($q) => $q->where('slug', $slug))
            ->latest('published_at')
            ->paginate(12);

        return Inertia::render('Blog/Tag', array_merge($this->getCommonData(), [
            'tag' => $tag,
            'posts' => $posts,
            'categories' => Category::withCount('publishedPosts')->get(),
        ]));
    }
}
