<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Notifications\WalletDepositConfirmed;
use App\Notifications\WalletTransactionRejected;
use App\Notifications\WalletWithdrawalApproved;
use Illuminate\Http\Request;
use Inertia\Inertia;

class WalletController extends Controller
{
    /**
     * Display listing of wallets with pending transactions.
     */
    public function index(Request $request)
    {
        $query = WalletTransaction::with(['wallet.user:id,name,email']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        } else {
            // Default to showing pending
            $query->pending();
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Search by user
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('wallet.user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $transactions = $query->latest()->paginate(20)->withQueryString();

        // Stats
        $stats = [
            'pending_deposits' => WalletTransaction::pending()
                ->where('type', 'deposit')
                ->count(),
            'pending_withdrawals' => WalletTransaction::pending()
                ->where('type', 'withdrawal')
                ->count(),
            'pending_amount' => WalletTransaction::pending()
                ->where('type', 'withdrawal')
                ->sum('amount'),
            'total_wallets' => Wallet::count(),
        ];

        return Inertia::render('Admin/Wallets/Index', [
            'transactions' => $transactions,
            'stats' => $stats,
            'filters' => $request->only(['search', 'status', 'type']),
        ]);
    }

    /**
     * Show a specific wallet with all transactions.
     */
    public function show(Wallet $wallet)
    {
        $wallet->load(['user:id,name,email,role']);
        
        $transactions = $wallet->transactions()
            ->latest()
            ->paginate(30);

        return Inertia::render('Admin/Wallets/Show', [
            'wallet' => $wallet,
            'transactions' => $transactions,
        ]);
    }

    /**
     * Approve a pending transaction (deposit or withdrawal).
     */
    public function approveTransaction(WalletTransaction $transaction)
    {
        if ($transaction->status !== WalletTransaction::STATUS_PENDING) {
            return back()->with('error', 'Esta transacción no está pendiente.');
        }

        try {
            $wallet = $transaction->wallet;

            if ($transaction->type === WalletTransaction::TYPE_DEPOSIT) {
                // Confirm deposit - add money to wallet
                $wallet->confirmTransaction($transaction);

                // Notify coach of confirmed deposit
                $wallet->user->notify(new WalletDepositConfirmed($transaction->fresh()));
            } elseif ($transaction->type === WalletTransaction::TYPE_WITHDRAWAL) {
                // Withdrawal - money already deducted, just mark as completed
                $transaction->update(['status' => WalletTransaction::STATUS_COMPLETED]);

                // Notify coach of approved withdrawal
                $wallet->user->notify(new WalletWithdrawalApproved($transaction));
            }

            return back()->with('success', 'Transacción aprobada correctamente.');
        } catch (\Exception $e) {
            return back()->with('error', 'Error al aprobar: ' . $e->getMessage());
        }
    }

    /**
     * Reject a pending transaction.
     */
    public function rejectTransaction(WalletTransaction $transaction, Request $request)
    {
        if ($transaction->status !== WalletTransaction::STATUS_PENDING) {
            return back()->with('error', 'Esta transacción no está pendiente.');
        }

        try {
            $wallet = $transaction->wallet;

            // If it was a withdrawal, refund the amount back to wallet
            if ($transaction->type === WalletTransaction::TYPE_WITHDRAWAL) {
                $wallet->increment('balance', abs($transaction->amount));
            }

            $transaction->update([
                'status' => WalletTransaction::STATUS_CANCELLED,
                'metadata' => array_merge($transaction->metadata ?? [], [
                    'rejection_reason' => $request->input('reason', 'Rechazado por administrador'),
                    'rejected_at' => now()->toISOString(),
                ]),
            ]);

            // Notify coach of rejected transaction
            $reason = $request->input('reason', 'Rechazado por administrador');
            $wallet->user->notify(new WalletTransactionRejected($transaction, $reason));

            return back()->with('success', 'Transacción rechazada.');
        } catch (\Exception $e) {
            return back()->with('error', 'Error al rechazar: ' . $e->getMessage());
        }
    }
}
