<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\User;
use App\Notifications\CoachApproved;
use App\Notifications\SubscriptionActivated;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Inertia\Inertia;
use Inertia\Response;

class UserController extends Controller
{
    /**
     * Display a listing of coaches.
     */
    public function index(Request $request): Response
    {
        $query = User::coaches()
            ->withCount('clients')
            ->with('activeSubscription.plan');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $users = $query->latest()->paginate(15)->withQueryString();

        // Add avatar_url to each user
        $users->getCollection()->transform(function ($user) {
            $user->avatar_url = $user->avatar 
                ? (str_starts_with($user->avatar, 'http') ? $user->avatar : '/storage/' . $user->avatar)
                : null;
            return $user;
        });

        return Inertia::render('Admin/Users/Index', [
            'users' => $users,
            'filters' => $request->only(['status', 'search']),
            'plans' => Plan::where('is_active', true)->get(['id', 'name', 'duration_days']),
        ]);
    }

    /**
     * Show the form for creating a new coach.
     */
    public function create(): Response
    {
        return Inertia::render('Admin/Users/Create', [
            'plans' => Plan::where('is_active', true)->get(['id', 'name', 'duration_days', 'price']),
        ]);
    }

    /**
     * Store a newly created coach.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => ['required', 'confirmed', Password::defaults()],
            'phone' => 'nullable|string|max:20',
            'status' => 'required|in:active,pending,suspended',
            'plan_id' => 'nullable|exists:plans,id',
            'plan_duration_days' => 'nullable|integer|min:1|max:365',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'phone' => $validated['phone'] ?? null,
            'role' => 'coach',
            'status' => $validated['status'],
            'email_verified_at' => now(),
        ]);

        // Assign plan if selected
        if (!empty($validated['plan_id'])) {
            $plan = Plan::find($validated['plan_id']);
            $duration = $validated['plan_duration_days'] ?? $plan->duration_days;
            
            Subscription::create([
                'user_id' => $user->id,
                'plan_id' => $plan->id,
                'start_date' => now(),
                'end_date' => now()->addDays($duration),
                'status' => 'active',
            ]);
        }

        return redirect()->route('admin.users.index')
            ->with('success', "Coach {$user->name} creado correctamente.");
    }

    /**
     * Display the specified user.
     */
    public function show(User $user): Response
    {
        return Inertia::render('Admin/Users/Show', [
            'user' => $user->load(['subscriptions.plan', 'invoices', 'clients']),
            'stats' => [
                'clients' => $user->clients()->count(),
                'workouts' => $user->clients()->withCount('workouts')->get()->sum('workouts_count'),
                'meal_plans' => $user->clients()->withCount('mealPlans')->get()->sum('meal_plans_count'),
            ],
            'plans' => Plan::where('is_active', true)->get(['id', 'name', 'duration_days']),
        ]);
    }

    /**
     * Show the form for editing the specified coach.
     */
    public function edit(User $user): Response
    {
        return Inertia::render('Admin/Users/Edit', [
            'user' => $user->load('activeSubscription.plan'),
            'plans' => Plan::where('is_active', true)->get(['id', 'name', 'duration_days', 'price']),
        ]);
    }

    /**
     * Update the specified coach.
     */
    public function update(Request $request, User $user): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => ['nullable', 'confirmed', Password::defaults()],
            'phone' => 'nullable|string|max:20',
            'status' => 'required|in:active,pending,suspended',
            'plan_id' => 'nullable|exists:plans,id',
            'plan_duration_days' => 'nullable|integer|min:1|max:365',
        ]);

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'status' => $validated['status'],
        ]);

        if (!empty($validated['password'])) {
            $user->update(['password' => Hash::make($validated['password'])]);
        }

        // Handle plan assignment/change
        if (!empty($validated['plan_id'])) {
            $plan = Plan::find($validated['plan_id']);
            $duration = $validated['plan_duration_days'] ?? $plan->duration_days;
            
            // Deactivate current subscription if exists
            $user->subscriptions()->where('status', 'active')->update(['status' => 'cancelled']);
            
            // Create new subscription
            Subscription::create([
                'user_id' => $user->id,
                'plan_id' => $plan->id,
                'start_date' => now(),
                'end_date' => now()->addDays($duration),
                'status' => 'active',
            ]);
        }

        return redirect()->route('admin.users.index')
            ->with('success', "Coach {$user->name} actualizado correctamente.");
    }

    /**
     * Approve a pending coach.
     */
    public function approve(User $user): RedirectResponse
    {
        if ($user->status !== 'pending') {
            return back()->with('error', 'Este usuario no está pendiente de aprobación.');
        }

        $user->update(['status' => 'active']);

        // Enviar notificación de aprobación
        $user->notify(new CoachApproved());

        return back()->with('success', "El usuario {$user->name} ha sido aprobado.");
    }

    /**
     * Suspend a coach.
     */
    public function suspend(User $user): RedirectResponse
    {
        if ($user->isAdmin()) {
            return back()->with('error', 'No se puede suspender a un administrador.');
        }

        $user->update(['status' => 'suspended']);

        // Also suspend their active subscription
        $user->subscriptions()->where('status', 'active')->update([
            'status' => 'suspended',
        ]);

        return back()->with('success', "El usuario {$user->name} ha sido suspendido.");
    }

    /**
     * Reactivate a suspended coach.
     */
    public function reactivate(User $user): RedirectResponse
    {
        if ($user->status !== 'suspended') {
            return back()->with('error', 'Este usuario no está suspendido.');
        }

        $user->update(['status' => 'active']);

        // Also reactivate their suspended subscription
        $user->subscriptions()->where('status', 'suspended')->update([
            'status' => 'active',
        ]);

        return back()->with('success', "El usuario {$user->name} ha sido reactivado.");
    }

    /**
     * Assign a plan to a user manually.
     */
    public function assignPlan(Request $request, User $user): RedirectResponse
    {
        $validated = $request->validate([
            'plan_id' => 'required|exists:plans,id',
            'duration_days' => 'required|integer|min:1|max:365',
        ]);

        $plan = Plan::findOrFail($validated['plan_id']);

        // Expire any existing active subscription
        Subscription::where('user_id', $user->id)
            ->where('status', 'active')
            ->update(['status' => 'expired']);

        // Create new subscription
        $subscription = Subscription::create([
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'start_date' => now(),
            'end_date' => now()->addDays($validated['duration_days']),
            'status' => 'active',
        ]);

        // Notificar al usuario sobre la activación
        $user->notify(new SubscriptionActivated($subscription));

        return back()->with('success', "Plan {$plan->name} asignado a {$user->name}.");
    }

    /**
     * Delete a user.
     */
    public function destroy(User $user): RedirectResponse
    {
        if ($user->isAdmin()) {
            return back()->with('error', 'No se puede eliminar a un administrador.');
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'Usuario eliminado correctamente.');
    }

    /**
     * Impersonate a user (login as them).
     */
    public function impersonate(User $user): RedirectResponse
    {
        if ($user->isAdmin()) {
            return back()->with('error', 'No se puede suplantar a un administrador.');
        }

        // Store original admin ID in session
        session(['impersonating_from' => Auth::id()]);
        
        // Login as the target user
        Auth::login($user);

        // Redirect to dashboard
        return redirect()->route('dashboard')
            ->with('success', "Ahora estás navegando como {$user->name}.");
    }

    /**
     * Stop impersonating and return to admin account.
     */
    public function stopImpersonating(): RedirectResponse
    {
        $originalUserId = session('impersonating_from');

        if (!$originalUserId) {
            return redirect()->route('dashboard');
        }

        // Get original admin user
        $originalUser = User::find($originalUserId);

        if (!$originalUser || !$originalUser->isAdmin()) {
            session()->forget('impersonating_from');
            return redirect()->route('dashboard')
                ->with('error', 'No se pudo restaurar la sesión original.');
        }

        // Clear impersonation session
        session()->forget('impersonating_from');

        // Login back as admin
        Auth::login($originalUser);

        return redirect()->route('admin.users.index')
            ->with('success', 'Has vuelto a tu cuenta de administrador.');
    }

    /**
     * Get clients for a specific coach.
     */
    public function clients(User $user): Response
    {
        return Inertia::render('Admin/Users/Clients', [
            'user' => $user,
            'clients' => $user->clients()
                ->withCount(['workouts', 'mealPlans'])
                ->latest()
                ->paginate(20),
        ]);
    }

    /**
     * Delete all invoices for a coach.
     */
    public function deleteInvoices(User $user): RedirectResponse
    {
        if ($user->isAdmin()) {
            return back()->with('error', 'No se pueden eliminar facturas de un administrador.');
        }

        $count = $user->invoices()->count();
        $user->invoices()->delete();

        return back()->with('success', "Se eliminaron {$count} facturas de {$user->name}.");
    }

    /**
     * Delete all messages for a coach.
     */
    public function deleteMessages(User $user): RedirectResponse
    {
        if ($user->isAdmin()) {
            return back()->with('error', 'No se pueden eliminar mensajes de un administrador.');
        }

        $sentCount = $user->sentMessages()->count();
        $receivedCount = $user->receivedMessages()->count();
        
        $user->sentMessages()->delete();
        $user->receivedMessages()->delete();

        $total = $sentCount + $receivedCount;
        return back()->with('success', "Se eliminaron {$total} mensajes de {$user->name}.");
    }

    /**
     * Delete all wallet transactions for a coach.
     */
    public function deleteTransactions(User $user): RedirectResponse
    {
        if ($user->isAdmin()) {
            return back()->with('error', 'No se pueden eliminar transacciones de un administrador.');
        }

        $wallet = $user->wallet;
        if (!$wallet) {
            return back()->with('error', 'El usuario no tiene billetera.');
        }

        $count = $wallet->transactions()->count();
        $wallet->transactions()->delete();
        
        // Reset wallet balance
        $wallet->update(['balance' => 0, 'pending_balance' => 0]);

        return back()->with('success', "Se eliminaron {$count} transacciones de {$user->name}.");
    }
}
