<?php

namespace App\Console\Commands;

use App\Models\Invoice;
use App\Models\User;
use App\Notifications\AdminInvoiceOverdue;
use App\Notifications\InvoiceOverdue;
use Carbon\Carbon;
use Illuminate\Console\Command;

class CheckOverdueInvoices extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoices:check-overdue {--days=7 : Days overdue to check}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for overdue invoices and notify users and admins';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $days = (int) $this->option('days');
        $this->info("Checking invoices overdue for {$days} days or more...");

        $overdueInvoices = Invoice::where('status', 'pending')
            ->where('created_at', '<=', Carbon::now()->subDays($days))
            ->with(['user'])
            ->get();

        $admins = User::where('role', 'admin')->get();

        $count = 0;
        foreach ($overdueInvoices as $invoice) {
            if ($invoice->user) {
                $daysPastDue = Carbon::now()->diffInDays($invoice->created_at);
                
                // Notify the coach (only once per week for overdue invoices)
                // Using a simple check: only notify if days past due is divisible by 7
                if ($daysPastDue % 7 === 0 || $daysPastDue === $days) {
                    $invoice->user->notify(new InvoiceOverdue($invoice, $daysPastDue));
                    $this->info("Notified {$invoice->user->name} - {$invoice->invoice_number} overdue {$daysPastDue} days");
                    
                    // Also notify admins
                    foreach ($admins as $admin) {
                        $admin->notify(new AdminInvoiceOverdue($invoice, $daysPastDue));
                    }
                    
                    $count++;
                }
            }
        }

        $this->info("Total: {$count} overdue invoice notifications sent.");

        return Command::SUCCESS;
    }
}
